<?php
// Veritabanı bağlantısını ve yapılandırmayı dahil et
require_once __DIR__ . '/../config/db_connect.php';

// Sayfa başlığını ayarla
$page_title = "Yeni Lisans Ekle";

// Header'ı dahil et (Oturum kontrolünü ve CSRF token'ı yapar)
require_once __DIR__ . '/includes/header.php'; // $csrf_token burada tanımlanır

$error_message = ''; // Form hataları için

// Form tekrar doldurma için varsayılan değerler
$product_id = '';
$customer_id = null; // Yeni eklendi
$domain = '';
$status = 'active'; // Varsayılan aktif
$expires_at_form = ''; // Form için datetime-local formatı
$client_name = ''; // Bu alanlar artık müşteri seçilince otomatik gelebilir veya kaldırılabilir
$client_email = '';
$notes = '';

// Ürünleri dropdown için çek
$products = [];
$product_sql = "SELECT id, name FROM products ORDER BY name ASC";
$product_result = $conn->query($product_sql);
if ($product_result && $product_result->num_rows > 0) {
    while ($row = $product_result->fetch_assoc()) {
        $products[] = $row;
    }
}

// Müşterileri dropdown için çek
$customers = [];
$customer_sql = "SELECT id, name, company FROM customers WHERE is_active = 1 ORDER BY name ASC";
$customer_result = $conn->query($customer_sql);
if ($customer_result && $customer_result->num_rows > 0) {
    while ($row = $customer_result->fetch_assoc()) {
        $customers[] = $row;
    }
}

// --- Yardımcı Fonksiyon: Benzersiz Lisans Anahtarı Üret ---
if (!function_exists('generate_unique_license_key')) {
    function generate_unique_license_key($conn, $prefix = 'LIC') {
        $max_attempts = 10;
        for ($i = 0; $i < $max_attempts; $i++) {
            $key = strtoupper($prefix . '-' . bin2hex(random_bytes(4)) . '-' . bin2hex(random_bytes(4)));
            $check_sql = "SELECT id FROM licenses WHERE license_key = ?";
            $stmt_check = $conn->prepare($check_sql);
            if ($stmt_check) {
                $stmt_check->bind_param("s", $key);
                $stmt_check->execute();
                $result_check = $stmt_check->get_result();
                $stmt_check->close();
                if ($result_check->num_rows === 0) { return $key; }
            } else { error_log("generate_unique_license_key - SQL prepare hatası: " . $conn->error); break; }
        }
        return null;
    }
}

// --- Form Gönderildi mi Kontrol Et (POST) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // CSRF Token Doğrulama
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $error_message = "Geçersiz veya eksik güvenlik token'ı.";
        // Hata durumunda formu tekrar göstermek için değişkenleri ayarla
        $product_id = (int)($_POST['product_id'] ?? 0);
        $customer_id = !empty($_POST['customer_id']) ? (int)$_POST['customer_id'] : null; // Müşteri ID'sini al
        $domain = trim($_POST['domain'] ?? '');
        $status = $_POST['status'] ?? 'inactive';
        $expires_at_form = trim($_POST['expires_at'] ?? '');
        $notes = trim($_POST['notes'] ?? '');
    } else {
        // Token geçerli, işlemlere devam et
        $product_id = (int)($_POST['product_id'] ?? 0);
        $customer_id = !empty($_POST['customer_id']) ? (int)$_POST['customer_id'] : null; // Müşteri ID'sini al
        $domain = trim($_POST['domain'] ?? '');
        $status = $_POST['status'] ?? 'inactive';
        $expires_at_input = trim($_POST['expires_at'] ?? '');
        $notes = trim($_POST['notes'] ?? '');
        // client_name ve client_email artık doğrudan kullanılmıyor, customer_id üzerinden alınabilir.

        $expires_at_form = $expires_at_input; // Formu tekrar doldurmak için

        // Girdi Doğrulama
        if ($product_id <= 0) {
            $error_message = "Lütfen geçerli bir ürün seçin.";
        } elseif (!in_array($status, ['active', 'inactive', 'expired'])) {
            $error_message = "Geçersiz lisans durumu seçildi.";
        } else {
            $expires_at_db = null;
            if (!empty($expires_at_input)) {
                try {
                    $date = new DateTime($expires_at_input);
                    $expires_at_db = $date->format('Y-m-d H:i:s');
                } catch (Exception $e) { $error_message = "Geçersiz son kullanma tarihi formatı."; }
            }

            // Hata yoksa devam et
            if (empty($error_message)) {
                $license_key = generate_unique_license_key($conn, 'LIC');

                if ($license_key === null) {
                    $error_message = "Benzersiz lisans anahtarı üretilemedi. Lütfen tekrar deneyin.";
                } else {
                    // Veritabanına ekleme işlemi (customer_id eklendi, client_name/email kaldırıldı)
                    $sql = "INSERT INTO licenses (product_id, customer_id, license_key, domain, status, expires_at, notes)
                            VALUES (?, ?, ?, ?, ?, ?, ?)"; // 7 parametre
                    $stmt = $conn->prepare($sql);

                    if ($stmt === false) {
                        error_log("Add License - SQL prepare hatası: " . $conn->error);
                        $error_message = "Lisans eklenirken bir SQL hatası oluştu.";
                    } else {
                        // bind_param güncellendi (i, i, s, s, s, s, s)
                        $stmt->bind_param("iisssss",
                            $product_id,
                            $customer_id, // integer veya null
                            $license_key,
                            $domain,
                            $status,
                            $expires_at_db, // string (tarih) veya null
                            $notes
                        );

                        if ($stmt->execute()) {
                            $_SESSION['success_message'] = "Lisans başarıyla oluşturuldu (Anahtar: {$license_key}).";
                            header("Location: licenses.php");
                            exit;
                        } else {
                            error_log("Add License - SQL execute hatası: " . $stmt->error);
                            if ($conn->errno == 1062) { $error_message = "Lisans anahtarı çakışması! Lütfen tekrar deneyin."; }
                            else { $error_message = "Lisans eklenirken bir veritabanı hatası oluştu."; }
                        }
                        $stmt->close();
                    } // stmt prepare else sonu
                } // license key null else sonu
            } // empty error message if sonu
        } // validation else sonu
    } // csrf else sonu
} // POST isteği sonu

?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2"><i class="bi bi-plus-square-fill"></i> <?php echo htmlspecialchars($page_title); ?></h1>
</div>

<?php if ($error_message): ?>
    <div class="alert alert-danger" role="alert">
        <?php echo htmlspecialchars($error_message); ?>
    </div>
<?php endif; ?>

<div class="card shadow-sm">
    <div class="card-header">
        Yeni Lisans Bilgileri
    </div>
    <div class="card-body">
        <form action="add_license.php" method="post">
            <!-- CSRF Token -->
            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">

            <div class="row g-3">
                <div class="col-md-6">
                    <label for="product_id" class="form-label">Ürün <span class="text-danger">*</span></label>
                    <select class="form-select" name="product_id" id="product_id" required>
                        <option value="">-- Ürün Seçin --</option>
                        <?php foreach ($products as $prod): ?>
                            <option value="<?php echo $prod['id']; ?>" <?php echo ($product_id == $prod['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($prod['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                 <div class="col-md-6">
                    <label for="customer_id" class="form-label">Müşteri (İsteğe Bağlı)</label>
                    <select class="form-select" name="customer_id" id="customer_id">
                        <option value="">-- Müşteri Seçin (Yoksa Boş Bırakın) --</option>
                        <?php foreach ($customers as $cust): ?>
                            <option value="<?php echo $cust['id']; ?>" <?php echo ($customer_id == $cust['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($cust['name']); ?> <?php echo !empty($cust['company']) ? '(' . htmlspecialchars($cust['company']) . ')' : ''; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                     <div class="form-text">Lisansı belirli bir müşteriyle ilişkilendirin.</div>
                </div>
                <div class="col-md-6">
                    <label for="domain" class="form-label">Domain</label>
                    <input type="text" class="form-control" name="domain" id="domain" value="<?php echo htmlspecialchars($domain); ?>" placeholder="ornekdomain.com">
                    <div class="form-text">Boş veya '*' tüm domainlerde geçerli olur. 'www.' olmadan girin.</div>
                </div>
                <div class="col-md-6">
                    <label for="status" class="form-label">Durum <span class="text-danger">*</span></label>
                    <select class="form-select" name="status" id="status" required>
                        <option value="active" <?php echo ($status === 'active') ? 'selected' : ''; ?>>Aktif</option>
                        <option value="inactive" <?php echo ($status === 'inactive') ? 'selected' : ''; ?>>Pasif</option>
                        <option value="expired" <?php echo ($status === 'expired') ? 'selected' : ''; ?>>Süresi Dolmuş</option>
                    </select>
                </div>
                 <div class="col-md-6">
                    <label for="expires_at" class="form-label">Son Kullanma Tarihi (İsteğe Bağlı)</label>
                    <input type="datetime-local" class="form-control" name="expires_at" id="expires_at" value="<?php echo htmlspecialchars($expires_at_form); ?>">
                    <div class="form-text">Boş bırakılırsa lisans süresiz olur.</div>
                </div>
                 <!-- client_name ve client_email alanları kaldırıldı, müşteri seçimi yeterli -->
                 <div class="col-12">
                    <label for="notes" class="form-label">Notlar (İsteğe Bağlı)</label>
                    <textarea class="form-control" name="notes" id="notes" rows="3"><?php echo htmlspecialchars($notes); ?></textarea>
                </div>
            </div>

            <hr class="my-4">

            <button type="submit" class="btn btn-success btn-lg"><i class="bi bi-check-circle-fill"></i> Lisansı Oluştur</button>
            <a href="licenses.php" class="btn btn-secondary btn-lg"><i class="bi bi-x-circle-fill"></i> İptal</a>
        </form>
    </div>
</div>

<?php
// Footer'ı dahil et
require_once __DIR__ . '/includes/footer.php';
?>
