<?php
// Veritabanı bağlantısını ve yapılandırmayı dahil et
require_once __DIR__ . '/../config/db_connect.php';

// Sayfa başlığını ayarla
$page_title = "Lisansı Düzenle";

// Header'ı dahil et (Oturum kontrolünü ve CSRF token'ı yapar)
require_once __DIR__ . '/includes/header.php'; // $csrf_token burada tanımlanır

$error_message = '';
$license_id = null;

// Form/Veritabanı değerleri için değişkenler
$product_id = '';
$customer_id = null; // Yeni eklendi
$license_key = '';
$domain = '';
$status = '';
$expires_at_form = '';
$notes = '';

// --- Lisans ID'sini Al ve Kontrol Et ---
if (!isset($_GET['id']) || !filter_var($_GET['id'], FILTER_VALIDATE_INT)) {
    $_SESSION['error_message'] = "Geçersiz lisans ID'si.";
    header("Location: licenses.php");
    exit;
}
$license_id = (int)$_GET['id'];

// Ürünleri dropdown için çek
$products = [];
$product_sql = "SELECT id, name FROM products ORDER BY name ASC";
$product_result = $conn->query($product_sql);
if ($product_result && $product_result->num_rows > 0) {
    while ($row = $product_result->fetch_assoc()) { $products[] = $row; }
}

// Müşterileri dropdown için çek
$customers = [];
$customer_sql = "SELECT id, name, company FROM customers WHERE is_active = 1 ORDER BY name ASC"; // Sadece aktif müşteriler
$customer_result = $conn->query($customer_sql);
if ($customer_result && $customer_result->num_rows > 0) {
    while ($row = $customer_result->fetch_assoc()) { $customers[] = $row; }
}

// --- POST İsteği İşleme (Form Gönderildiğinde) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF Token Doğrulama
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
         $_SESSION['error_message'] = "Geçersiz güvenlik token'ı nedeniyle işlem yapılamadı.";
         header("Location: licenses.php");
         exit;
    }

    $posted_id = (int)($_POST['license_id'] ?? 0);
    if ($posted_id !== $license_id) {
        $_SESSION['error_message'] = "ID uyuşmazlığı nedeniyle işlem yapılamadı.";
        header("Location: licenses.php");
        exit;
    }

    // Form verilerini al
    $product_id = (int)($_POST['product_id'] ?? 0);
    $customer_id = !empty($_POST['customer_id']) ? (int)$_POST['customer_id'] : null; // Müşteri ID'sini al
    $domain = trim($_POST['domain'] ?? '');
    $status = $_POST['status'] ?? 'inactive';
    $expires_at_input = trim($_POST['expires_at'] ?? '');
    $notes = trim($_POST['notes'] ?? '');
    $license_key = trim($_POST['license_key_display'] ?? ''); // Sadece hata durumunda göstermek için

    $expires_at_form = $expires_at_input; // Formu tekrar doldurmak için

    // Girdi Doğrulama
    if ($product_id <= 0) {
        $error_message = "Lütfen geçerli bir ürün seçin.";
    } elseif (!in_array($status, ['active', 'inactive', 'expired'])) {
        $error_message = "Geçersiz lisans durumu seçildi.";
    } else {
        $expires_at_db = null;
        if (!empty($expires_at_input)) {
            try {
                $date = new DateTime($expires_at_input);
                $expires_at_db = $date->format('Y-m-d H:i:s');
            } catch (Exception $e) { $error_message = "Geçersiz son kullanma tarihi formatı."; }
        }

        // Hata yoksa güncelle
        if (empty($error_message)) {
            // client_name ve client_email kaldırıldı, customer_id eklendi
            $sql = "UPDATE licenses SET
                        product_id = ?, customer_id = ?, domain = ?, status = ?,
                        expires_at = ?, notes = ?
                    WHERE id = ?"; // 7 parametre
            $stmt = $conn->prepare($sql);

            if ($stmt === false) {
                error_log("Edit License - SQL prepare hatası: " . $conn->error);
                $error_message = "Lisans güncellenirken bir SQL hatası oluştu.";
            } else {
                // bind_param güncellendi (i, i, s, s, s, s, i)
                $stmt->bind_param("iissssi",
                    $product_id,
                    $customer_id, // integer veya null
                    $domain,
                    $status,
                    $expires_at_db, // string (tarih) veya null
                    $notes,
                    $license_id
                );

                if ($stmt->execute()) {
                    $_SESSION['success_message'] = "Lisans (ID: {$license_id}) başarıyla güncellendi.";
                    header("Location: licenses.php");
                    exit;
                } else {
                    error_log("Edit License - SQL execute hatası: " . $stmt->error);
                    $error_message = "Lisans güncellenirken bir veritabanı hatası oluştu.";
                }
                $stmt->close();
            }
        }
    } // Validation else sonu
} else {
    // --- GET İsteği İşleme (Sayfa İlk Yüklendiğinde) ---
    // customer_id'yi de çekelim
    $sql_get = "SELECT product_id, customer_id, license_key, domain, status, expires_at, notes
                FROM licenses WHERE id = ?";
    $stmt_get = $conn->prepare($sql_get);

    if ($stmt_get === false) {
        error_log("Edit License (GET) - SQL prepare hatası: " . $conn->error);
        $_SESSION['error_message'] = "Lisans bilgileri alınırken bir hata oluştu.";
        header("Location: licenses.php");
        exit;
    }

    $stmt_get->bind_param("i", $license_id);
    $stmt_get->execute();
    $result_get = $stmt_get->get_result();

    if ($result_get->num_rows === 1) {
        $license = $result_get->fetch_assoc();
        $product_id = $license['product_id'];
        $customer_id = $license['customer_id']; // Müşteri ID'sini al
        $license_key = $license['license_key'];
        $domain = $license['domain'] ?? '';
        $status = $license['status'];
        $expires_at_form = $license['expires_at'] ? date('Y-m-d\TH:i', strtotime($license['expires_at'])) : '';
        $notes = $license['notes'] ?? '';
        // client_name ve client_email artık çekilmiyor
    } else {
        $_SESSION['error_message'] = "Düzenlenecek lisans bulunamadı.";
        header("Location: licenses.php");
        exit;
    }
    $stmt_get->close();
}

?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
     <h1 class="h2"><i class="bi bi-pencil-square"></i> <?php echo htmlspecialchars($page_title); ?> (ID: <?php echo $license_id; ?>)</h1>
</div>

<?php if ($error_message): ?>
    <div class="alert alert-danger" role="alert">
        <?php echo htmlspecialchars($error_message); ?>
    </div>
<?php endif; ?>

<div class="card shadow-sm">
    <div class="card-header">
        Lisans Bilgilerini Düzenle
    </div>
    <div class="card-body">
        <form action="edit_license.php?id=<?php echo $license_id; ?>" method="post">
            <input type="hidden" name="license_id" value="<?php echo $license_id; ?>">
            <input type="hidden" name="license_key_display" value="<?php echo htmlspecialchars($license_key); ?>">
            <!-- CSRF Token -->
            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">

            <div class="row g-3">
                 <div class="col-12">
                    <label for="license_key_display" class="form-label">Lisans Anahtarı</label>
                    <input type="text" class="form-control" id="license_key_display" value="<?php echo htmlspecialchars($license_key); ?>" readonly disabled>
                    <div class="form-text">Lisans anahtarı değiştirilemez.</div>
                </div>

                <div class="col-md-6">
                    <label for="product_id" class="form-label">Ürün <span class="text-danger">*</span></label>
                    <select class="form-select" name="product_id" id="product_id" required>
                        <option value="">-- Ürün Seçin --</option>
                        <?php foreach ($products as $prod): ?>
                            <option value="<?php echo $prod['id']; ?>" <?php echo ($product_id == $prod['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($prod['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                 <div class="col-md-6">
                    <label for="customer_id" class="form-label">Müşteri (İsteğe Bağlı)</label>
                    <select class="form-select" name="customer_id" id="customer_id">
                        <option value="">-- Müşteri Seçin (Yoksa Boş Bırakın) --</option>
                        <?php foreach ($customers as $cust): ?>
                            <option value="<?php echo $cust['id']; ?>" <?php echo ($customer_id == $cust['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($cust['name']); ?> <?php echo !empty($cust['company']) ? '(' . htmlspecialchars($cust['company']) . ')' : ''; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                     <div class="form-text">Lisansı belirli bir müşteriyle ilişkilendirin.</div>
                </div>
                <div class="col-md-6">
                    <label for="domain" class="form-label">Domain</label>
                    <input type="text" class="form-control" name="domain" id="domain" value="<?php echo htmlspecialchars($domain); ?>" placeholder="ornekdomain.com">
                    <div class="form-text">Boş veya '*' tüm domainlerde geçerli olur. 'www.' olmadan girin.</div>
                </div>
                <div class="col-md-6">
                    <label for="status" class="form-label">Durum <span class="text-danger">*</span></label>
                    <select class="form-select" name="status" id="status" required>
                        <option value="active" <?php echo ($status === 'active') ? 'selected' : ''; ?>>Aktif</option>
                        <option value="inactive" <?php echo ($status === 'inactive') ? 'selected' : ''; ?>>Pasif</option>
                        <option value="expired" <?php echo ($status === 'expired') ? 'selected' : ''; ?>>Süresi Dolmuş</option>
                    </select>
                </div>
                 <div class="col-md-6">
                    <label for="expires_at" class="form-label">Son Kullanma Tarihi (İsteğe Bağlı)</label>
                    <input type="datetime-local" class="form-control" name="expires_at" id="expires_at" value="<?php echo htmlspecialchars($expires_at_form); ?>">
                    <div class="form-text">Boş bırakılırsa lisans süresiz olur.</div>
                </div>
                 <!-- client_name ve client_email alanları kaldırıldı -->
                 <div class="col-12">
                    <label for="notes" class="form-label">Notlar (İsteğe Bağlı)</label>
                    <textarea class="form-control" name="notes" id="notes" rows="3"><?php echo htmlspecialchars($notes); ?></textarea>
                </div>
            </div>

            <hr class="my-4">

            <button type="submit" class="btn btn-primary btn-lg"><i class="bi bi-save-fill"></i> Değişiklikleri Kaydet</button>
            <a href="licenses.php" class="btn btn-secondary btn-lg"><i class="bi bi-x-circle-fill"></i> İptal</a>
        </form>
    </div>
</div>

<?php
// Footer'ı dahil et
require_once __DIR__ . '/includes/footer.php';
?>
