<?php
/**
 * Dijital Menü - Masalar Sayfası
 */

// İstanbul saatini ayarla
date_default_timezone_set('Europe/Istanbul');

// Gerekli dosyaları dahil et
require_once '../includes/auth.php';
require_once '../includes/language.php';

// Admin paneli dilini al
$adminLanguage = getAdminLanguage();

// Giriş yapılmamışsa yönlendir
redirectIfNotLoggedIn('login.php');

// Admin yetkisi kontrolü
requirePermission('admin', 'login.php');

// Sayfa başlığı
$pageTitle = __('admin.tables', [], $adminLanguage);

// Ayarları al
$settings = getSetting();

// Masa işlemleri
$action = isset($_GET['action']) ? $_GET['action'] : 'list';
$tableId = isset($_GET['id']) ? intval($_GET['id']) : 0;
$error = '';
$success = '';

// Masa silme işlemi
if (isset($_GET['delete']) && $tableId > 0) {
    // Masanın kullanımda olup olmadığını kontrol et
    $result = $conn->query("SELECT COUNT(*) as count FROM waiter_calls WHERE table_id = $tableId AND status = 'pending'");
    $row = $result->fetch_assoc();
    
    if ($row['count'] > 0) {
        $_SESSION['error_message'] = 'Bu masa için bekleyen garson çağrıları olduğu için silinemez.';
    } else {
        // Masayı sil
        $stmt = $conn->prepare("DELETE FROM tables WHERE id = ?");
        $stmt->bind_param("i", $tableId);
        
        if ($stmt->execute()) {
            // Tamamlanmış garson çağrılarını da sil
            $conn->query("DELETE FROM waiter_calls WHERE table_id = $tableId");
            
            $_SESSION['success_message'] = 'Masa başarıyla silindi.';
        } else {
            $_SESSION['error_message'] = 'Masa silinirken bir hata oluştu.';
        }
    }
    
    redirect('tables.php');
}

// Masa ekleme/düzenleme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Form verilerini al
    $name = isset($_POST['name']) ? clean($_POST['name']) : '';
    $capacity = isset($_POST['capacity']) ? intval($_POST['capacity']) : 0;
    $location = isset($_POST['location']) ? clean($_POST['location']) : '';
    $status = isset($_POST['status']) ? clean($_POST['status']) : 'active';
    
    // QR kod oluştur
    $qrCode = '';
    if ($action === 'add') {
        // Yeni masa için QR kod oluştur
        $qrCode = uniqid() . '.png';
    } elseif ($action === 'edit' && isset($_POST['current_qr_code']) && !empty($_POST['current_qr_code'])) {
        $qrCode = $_POST['current_qr_code'];
    }
    
    // Boş alan kontrolü
    if (empty($name)) {
        $error = 'Masa adı alanı boş bırakılamaz.';
    } else {
        if ($action === 'add') {
            // Masa ekle
            $stmt = $conn->prepare("INSERT INTO tables (name, capacity, location, status, qr_code) VALUES (?, ?, ?, ?, ?)");
            $stmt->bind_param("sisss", $name, $capacity, $location, $status, $qrCode);
            
            if ($stmt->execute()) {
                $newTableId = $conn->insert_id;
                
                // QR kod oluştur
                $qrFileName = 'table_' . $newTableId . '_' . uniqid() . '.png';
                $qrCode = generateQRCode($newTableId, $qrFileName, '../assets/uploads/qrcodes');
                
                // Site URL'sini ayarlardan al, yoksa varsayılan değeri kullan
                if (!isset($settings['site_url']) || empty($settings['site_url'])) {
                    // Kullanıcıya site URL'sini ayarlaması gerektiğini hatırlat
                    $_SESSION['warning_message'] = 'QR kod oluşturuldu, ancak site URL\'si ayarlanmamış. Lütfen Ayarlar > Diğer Ayarlar bölümünden site URL\'sini ayarlayın.';
                }
                
                // QR kod dosya adını güncelle
                if ($qrCode) {
                    $stmt = $conn->prepare("UPDATE tables SET qr_code = ? WHERE id = ?");
                    $stmt->bind_param("si", $qrCode, $newTableId);
                    $stmt->execute();
                }
                
                $_SESSION['success_message'] = 'Masa başarıyla eklendi.';
                redirect('tables.php');
            } else {
                $error = 'Masa eklenirken bir hata oluştu.';
            }
        } elseif ($action === 'edit' && $tableId > 0) {
            // Masa güncelle
            $stmt = $conn->prepare("UPDATE tables SET name = ?, capacity = ?, location = ?, status = ?, qr_code = ? WHERE id = ?");
            $stmt->bind_param("sisssi", $name, $capacity, $location, $status, $qrCode, $tableId);
            
            if ($stmt->execute()) {
                // QR kod oluştur veya güncelle
                $qrFileName = 'table_' . $tableId . '_' . uniqid() . '.png';
                $qrCode = generateQRCode($tableId, $qrFileName, '../assets/uploads/qrcodes');
                
                // QR kod dosya adını güncelle
                if ($qrCode) {
                    $stmt = $conn->prepare("UPDATE tables SET qr_code = ? WHERE id = ?");
                    $stmt->bind_param("si", $qrCode, $tableId);
                    $stmt->execute();
                }
                
                $_SESSION['success_message'] = 'Masa başarıyla güncellendi.';
                redirect('tables.php');
            } else {
                $error = 'Masa güncellenirken bir hata oluştu.';
            }
        }
    }
}

// Masa bilgilerini al
$table = [];
if ($action === 'edit' && $tableId > 0) {
    $stmt = $conn->prepare("SELECT * FROM tables WHERE id = ?");
    $stmt->bind_param("i", $tableId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        $table = $result->fetch_assoc();
    } else {
        $_SESSION['error_message'] = 'Masa bulunamadı.';
        redirect('tables.php');
    }
}

// Tüm masaları al
$tables = getTables();

// Aktif kullanıcı bilgilerini al
$user = getUser($_SESSION['user_id']);
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?> | <?php echo $settings['site_title'] ?? 'Dijital Menü'; ?></title>
    
    <!-- Favicon -->
    <?php if (!empty($settings['favicon'])): ?>
    <link rel="shortcut icon" href="../assets/img/<?php echo $settings['favicon']; ?>" type="image/x-icon">
    <?php endif; ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    
    <!-- Ana CSS -->
    <link href="../assets/css/style.css" rel="stylesheet">
    
    <!-- Özel Renkler -->
    <style>
        :root {
            --primary-color: <?php echo $settings['primary_color'] ?? '#3490dc'; ?>;
            --secondary-color: <?php echo $settings['secondary_color'] ?? '#38c172'; ?>;
            --text-color: <?php echo $settings['text_color'] ?? '#333333'; ?>;
            --background-color: <?php echo $settings['background_color'] ?? '#ffffff'; ?>;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include 'includes/sidebar.php'; ?>
            
            <!-- Ana İçerik -->
            <div class="col-md-9 col-lg-10 admin-content">
                <!-- Üst Başlık -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h1 class="h3"><?php echo $pageTitle; ?></h1>
                    <?php if ($action === 'list'): ?>
                    <a href="tables.php?action=add" class="btn btn-primary">
                        <i class="fas fa-plus-circle"></i> <?php echo __('admin.add_table', [], $adminLanguage); ?>
                    </a>
                    <?php else: ?>
                    <a href="tables.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> <?php echo __('admin.go_back', [], $adminLanguage); ?>
                    </a>
                    <?php endif; ?>
                </div>
                
                <!-- Oturum Mesajı -->
                <?php if (isset($_SESSION['error_message'])): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?php echo $_SESSION['error_message']; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['error_message']); ?>
                <?php endif; ?>
                
                <?php if (isset($_SESSION['success_message'])): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <?php echo $_SESSION['success_message']; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['success_message']); ?>
                <?php endif; ?>
                
                <?php if (!empty($error)): ?>
                <div class="alert alert-danger"><?php echo $error; ?></div>
                <?php endif; ?>
                
                <?php if (!empty($success)): ?>
                <div class="alert alert-success"><?php echo $success; ?></div>
                <?php endif; ?>
                
                <!-- İçerik -->
                <div class="admin-card">
                    <?php if ($action === 'list'): ?>
                    <!-- Masa Listesi -->
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th class="d-none">ID</th>
                                    <th><?php echo __('admin.name', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.capacity', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.location', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.qr_code', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.status', [], $adminLanguage); ?></th>
                                    <th><?php echo __('admin.actions', [], $adminLanguage); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($tables)): ?>
                                <tr>
                                    <td colspan="7" class="text-center"><?php echo __('admin.no_tables', [], $adminLanguage); ?></td>
                                </tr>
                                <?php else: ?>
                                <?php foreach ($tables as $table): ?>
                                <tr>
                                    <td class="d-none"><?php echo $table['id']; ?></td>
                                    <td><?php echo $table['name']; ?></td>
                                    <td><?php echo isset($table['capacity']) ? $table['capacity'] : '0'; ?> <?php echo __('admin.people', [], $adminLanguage); ?></td>
                                    <td><?php echo isset($table['location']) ? $table['location'] : ''; ?></td>
                                    <td>
                                        <?php if (!empty($table['qr_code'])): ?>
                                        <div class="qr-code-container">
                                            <?php if (strpos($table['qr_code'], '.html') !== false): ?>
                                            <a href="../assets/uploads/qrcodes/<?php echo $table['qr_code']; ?>" target="_blank" class="btn btn-sm btn-primary">
                                                <i class="fas fa-qrcode"></i> <?php echo __('admin.view_qr', [], $adminLanguage); ?>
                                            </a>
                                            <?php else: ?>
                                            <img src="../assets/uploads/qrcodes/<?php echo $table['qr_code']; ?>" alt="QR Kod" width="50" height="50" class="qr-code-image">
                                            <a href="../assets/uploads/qrcodes/<?php echo $table['qr_code']; ?>" download="<?php echo strtolower(str_replace(' ', '-', $table['name'])); ?>-qr.png" class="btn btn-sm btn-primary ms-2">
                                                <i class="fas fa-download"></i>
                                            </a>
                                            <?php endif; ?>
                                        </div>
                                        <?php else: ?>
                                        <span class="text-muted"><?php echo __('admin.no_qr_code', [], $adminLanguage); ?></span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($table['status'] === 'active'): ?>
                                        <span class="badge bg-success"><?php echo __('admin.active', [], $adminLanguage); ?></span>
                                        <?php elseif ($table['status'] === 'inactive'): ?>
                                        <span class="badge bg-danger"><?php echo __('admin.inactive', [], $adminLanguage); ?></span>
                                        <?php elseif ($table['status'] === 'reserved'): ?>
                                        <span class="badge bg-warning"><?php echo __('admin.reserved', [], $adminLanguage); ?></span>
                                        <?php elseif ($table['status'] === 'occupied'): ?>
                                        <span class="badge bg-info"><?php echo __('admin.occupied', [], $adminLanguage); ?></span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <a href="tables.php?action=edit&id=<?php echo $table['id']; ?>" class="btn btn-sm btn-primary">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <a href="tables.php?delete=1&id=<?php echo $table['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('<?php echo __('admin.confirm_delete', [], $adminLanguage); ?>');">
                                            <i class="fas fa-trash"></i>
                                        </a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php else: ?>
                    <!-- Masa Formu -->
                    <form method="post" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF'] . '?action=' . $action . ($tableId ? '&id=' . $tableId : '')); ?>">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="name" class="form-label"><?php echo __('admin.name', [], $adminLanguage); ?> <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="name" name="name" value="<?php echo $table['name'] ?? ''; ?>" required>
                                    <small class="text-muted"><?php echo __('admin.table_name_example', [], $adminLanguage); ?></small>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="capacity" class="form-label"><?php echo __('admin.capacity', [], $adminLanguage); ?></label>
                                    <input type="number" class="form-control" id="capacity" name="capacity" value="<?php echo $table['capacity'] ?? ''; ?>" min="1">
                                    <small class="text-muted"><?php echo __('admin.capacity_help', [], $adminLanguage); ?></small>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="location" class="form-label"><?php echo __('admin.location', [], $adminLanguage); ?></label>
                                    <input type="text" class="form-control" id="location" name="location" value="<?php echo $table['location'] ?? ''; ?>">
                                    <small class="text-muted"><?php echo __('admin.location_example', [], $adminLanguage); ?></small>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="status" class="form-label"><?php echo __('admin.status', [], $adminLanguage); ?></label>
                                    <select class="form-select" id="status" name="status">
                                        <option value="active" <?php echo (isset($table['status']) && $table['status'] === 'active') ? 'selected' : ''; ?>><?php echo __('admin.active', [], $adminLanguage); ?></option>
                                        <option value="inactive" <?php echo (isset($table['status']) && $table['status'] === 'inactive') ? 'selected' : ''; ?>><?php echo __('admin.inactive', [], $adminLanguage); ?></option>
                                        <option value="reserved" <?php echo (isset($table['status']) && $table['status'] === 'reserved') ? 'selected' : ''; ?>><?php echo __('admin.reserved', [], $adminLanguage); ?></option>
                                        <option value="occupied" <?php echo (isset($table['status']) && $table['status'] === 'occupied') ? 'selected' : ''; ?>><?php echo __('admin.occupied', [], $adminLanguage); ?></option>
                                    </select>
                                </div>
                                
                                <?php if ($action === 'edit' && !empty($table['qr_code'])): ?>
                                <div class="mb-3">
                                    <label class="form-label"><?php echo __('admin.qr_code', [], $adminLanguage); ?></label>
                                    <div class="qr-code-container">
                                        <?php if (strpos($table['qr_code'], '.html') !== false): ?>
                                        <a href="../assets/uploads/qrcodes/<?php echo $table['qr_code']; ?>" target="_blank" class="btn btn-sm btn-primary">
                                            <i class="fas fa-qrcode"></i> <?php echo __('admin.view_qr', [], $adminLanguage); ?>
                                        </a>
                                        <?php else: ?>
                                        <img src="../assets/uploads/qrcodes/<?php echo $table['qr_code']; ?>" alt="QR Kod" class="qr-code-image">
                                        <div class="mt-2">
                                            <a href="../assets/uploads/qrcodes/<?php echo $table['qr_code']; ?>" download="<?php echo strtolower(str_replace(' ', '-', $table['name'])); ?>-qr.png" class="btn btn-sm btn-primary">
                                                <i class="fas fa-download"></i> <?php echo __('admin.download_qr', [], $adminLanguage); ?>
                                            </a>
                                        </div>
                                        <?php endif; ?>
                                        <input type="hidden" name="current_qr_code" value="<?php echo $table['qr_code']; ?>">
                                    </div>
                                    <small class="text-muted"><?php echo __('admin.qr_code_help', [], $adminLanguage); ?></small>
                                </div>
                                <?php endif; ?>
                                
                                <div class="mb-3">
                                    <label class="form-label"><?php echo __('admin.table_url', [], $adminLanguage); ?></label>
                                    <div class="input-group">
                                        <?php 
                                        $encodedTableId = '';
                                        if ($tableId) {
                                            $encodedTableId = base64_encode("Table:$tableId");
                                        }
                                        ?>
                                        <input type="text" class="form-control" value="<?php echo $settings['site_url'] ?? 'http://localhost/digital-menu'; ?>/index.php?qr=<?php echo $encodedTableId ?: '[ENCODED_ID]'; ?>" readonly>
                                        <button class="btn btn-outline-secondary" type="button" onclick="copyToClipboard(this.previousElementSibling)">
                                            <i class="fas fa-copy"></i>
                                        </button>
                                    </div>
                                    <small class="text-muted">Bu URL, masa bilgisini kodlanmış olarak içerir ve sadece QR kod ile erişildiğinde garson çağırma butonu görünür.</small>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mt-4">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> <?php echo __('admin.save', [], $adminLanguage); ?>
                            </button>
                            <a href="tables.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i> <?php echo __('admin.cancel', [], $adminLanguage); ?>
                            </a>
                        </div>
                    </form>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <!-- Ana JS -->
    <script src="../assets/js/script.js"></script>
    
    <!-- Yardımcı Fonksiyonlar -->
    <script>
        function copyToClipboard(element) {
            element.select();
            document.execCommand('copy');
            
            // Kopyalandı bildirimi
            const tooltip = document.createElement('div');
            tooltip.className = 'copy-tooltip';
            tooltip.textContent = '<?php echo __('admin.copied', [], $adminLanguage); ?>';
            document.body.appendChild(tooltip);
            
            setTimeout(() => {
                document.body.removeChild(tooltip);
            }, 2000);
        }
        
        function downloadQRCode(imgElement, fileName) {
            // QR kod resmini al
            const img = imgElement.tagName === 'IMG' ? imgElement : imgElement.querySelector('img');
            if (!img) return;
            
            // Canvas oluştur
            const canvas = document.createElement('canvas');
            canvas.width = img.naturalWidth || 200;
            canvas.height = img.naturalHeight || 200;
            
            // Resmi canvas'a çiz
            const ctx = canvas.getContext('2d');
            ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
            
            // Canvas'ı veri URL'sine dönüştür
            const dataURL = canvas.toDataURL('image/png');
            
            // İndirme bağlantısı oluştur
            const link = document.createElement('a');
            link.href = dataURL;
            link.download = fileName || 'qrcode.png';
            link.style.display = 'none';
            document.body.appendChild(link);
            
            // İndirme işlemini başlat
            link.click();
            
            // Bağlantıyı temizle
            setTimeout(() => {
                document.body.removeChild(link);
            }, 100);
        }
    </script>
</body>
</html>
