<?php
/**
 * Dijital Menü - Kullanıcı Ekleme Sayfası
 */

// İstanbul saatini ayarla
date_default_timezone_set('Europe/Istanbul');

// Veritabanı bağlantısını dahil et
require_once '../config/database.php';

// Gerekli dosyaları dahil et
require_once '../includes/auth.php';
require_once '../includes/functions.php';
require_once '../includes/language.php';

// Admin paneli dilini al
$adminLanguage = getAdminLanguage();

// Giriş yapılmamışsa yönlendir
redirectIfNotLoggedIn('login.php');

// Admin yetkisi kontrolü
requirePermission('admin', 'login.php');

// Sayfa başlığı
$pageTitle = __('admin.add_user', [], $adminLanguage);

// Ayarları al
$settings = getSetting();

// Hata ve başarı mesajları
$error = '';
$success = '';

// Kullanıcı ekleme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Form verilerini al
    $username = isset($_POST['username']) ? clean($_POST['username']) : '';
    $email = isset($_POST['email']) ? clean($_POST['email']) : '';
    $name = isset($_POST['name']) ? clean($_POST['name']) : '';
    $role = isset($_POST['role']) ? clean($_POST['role']) : 'waiter';
    $status = isset($_POST['status']) ? 1 : 0;
    $password = isset($_POST['password']) ? $_POST['password'] : '';
    $confirmPassword = isset($_POST['confirm_password']) ? $_POST['confirm_password'] : '';
    
    // Boş alan kontrolü
    if (empty($username) || empty($email) || empty($name) || empty($password)) {
        $error = 'Kullanıcı adı, e-posta, isim ve şifre alanları boş bırakılamaz.';
    } elseif ($password !== $confirmPassword) {
        $error = 'Şifreler eşleşmiyor.';
    } else {
        // E-posta formatı kontrolü
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = 'Geçerli bir e-posta adresi giriniz.';
        } else {
            // Kullanıcı adı ve e-posta benzersizlik kontrolü
            $stmt = $conn->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
            $stmt->bind_param("ss", $username, $email);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $error = 'Bu kullanıcı adı veya e-posta adresi zaten kullanılıyor.';
            } else {
                // Şifreyi hashle
                $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                
                // Kullanıcı ekle
                $stmt = $conn->prepare("INSERT INTO users (username, password, email, name, role, status) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->bind_param("sssssi", $username, $hashedPassword, $email, $name, $role, $status);
                
                if ($stmt->execute()) {
                    $_SESSION['success_message'] = 'Kullanıcı başarıyla eklendi.';
                    redirect('users.php');
                } else {
                    $error = 'Kullanıcı eklenirken bir hata oluştu: ' . $conn->error;
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?> | <?php echo $settings['site_title'] ?? 'Dijital Menü'; ?></title>
    
    <!-- Favicon -->
    <?php if (!empty($settings['favicon'])): ?>
    <link rel="shortcut icon" href="../assets/img/<?php echo $settings['favicon']; ?>" type="image/x-icon">
    <?php endif; ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    
    <!-- Ana CSS -->
    <link href="../assets/css/style.css" rel="stylesheet">
    <link href="../assets/css/admin.css" rel="stylesheet">
    
    <!-- Özel Renkler -->
    <style>
        :root {
            --primary-color: <?php echo $settings['primary_color'] ?? '#3490dc'; ?>;
            --secondary-color: <?php echo $settings['secondary_color'] ?? '#38c172'; ?>;
            --text-color: <?php echo $settings['text_color'] ?? '#333333'; ?>;
            --background-color: <?php echo $settings['background_color'] ?? '#ffffff'; ?>;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include 'includes/sidebar.php'; ?>
            
            <!-- Ana İçerik -->
            <div class="col-md-9 col-lg-10 admin-content">
                <!-- Üst Başlık -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h1 class="h3"><?php echo $pageTitle; ?></h1>
                    <a href="users.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> <?php echo __('admin.go_back', [], $adminLanguage); ?>
                    </a>
                </div>
                
                <!-- Oturum Mesajı -->
                <?php if (!empty($error)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?php echo $error; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($success)): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <?php echo $success; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php endif; ?>
                
                <!-- İçerik -->
                <div class="admin-card">
                    <!-- Kullanıcı Formu -->
                    <form method="post" action="user-add.php">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="username" class="form-label"><?php echo __('admin.username', [], $adminLanguage); ?> <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="username" name="username" value="<?php echo isset($_POST['username']) ? htmlspecialchars($_POST['username']) : ''; ?>" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="email" class="form-label"><?php echo __('admin.email', [], $adminLanguage); ?> <span class="text-danger">*</span></label>
                                    <input type="email" class="form-control" id="email" name="email" value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="name" class="form-label"><?php echo __('admin.name', [], $adminLanguage); ?> <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="name" name="name" value="<?php echo isset($_POST['name']) ? htmlspecialchars($_POST['name']) : ''; ?>" required>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="password" class="form-label"><?php echo __('admin.password', [], $adminLanguage); ?> <span class="text-danger">*</span></label>
                                    <input type="password" class="form-control" id="password" name="password" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="confirm_password" class="form-label"><?php echo __('admin.confirm_password', [], $adminLanguage); ?> <span class="text-danger">*</span></label>
                                    <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="role" class="form-label"><?php echo __('admin.role', [], $adminLanguage); ?></label>
                                    <select class="form-select" id="role" name="role">
                                        <option value="admin" <?php echo (isset($_POST['role']) && $_POST['role'] == 'admin') ? 'selected' : ''; ?>><?php echo __('admin.role_admin', [], $adminLanguage); ?></option>
                                        <option value="manager" <?php echo (isset($_POST['role']) && $_POST['role'] == 'manager') ? 'selected' : ''; ?>><?php echo __('admin.role_manager', [], $adminLanguage); ?></option>
                                        <option value="waiter" <?php echo (isset($_POST['role']) && $_POST['role'] == 'waiter') ? 'selected' : (isset($_POST['role']) ? '' : 'selected'); ?>><?php echo __('admin.role_waiter', [], $adminLanguage); ?></option>
                                    </select>
                                </div>
                                
                                <div class="mb-3 form-check">
                                    <input type="checkbox" class="form-check-input" id="status" name="status" <?php echo (isset($_POST['status'])) ? 'checked' : 'checked'; ?>>
                                    <label class="form-check-label" for="status"><?php echo __('admin.active', [], $adminLanguage); ?></label>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mt-4">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> <?php echo __('admin.save', [], $adminLanguage); ?>
                            </button>
                            <a href="users.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i> <?php echo __('admin.cancel', [], $adminLanguage); ?>
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <!-- Ana JS -->
    <script src="../assets/js/script.js"></script>
    <script src="../assets/js/admin.js"></script>
</body>
</html>
