<?php
/**
 * Dijital Menü - Garson Çağrıları Sayfası
 */

// İstanbul saatini ayarla
date_default_timezone_set('Europe/Istanbul');

// Gerekli dosyaları dahil et
require_once '../includes/auth.php';
require_once '../includes/language.php';

// Admin paneli dilini al
$adminLanguage = getAdminLanguage();

// Giriş yapılmamışsa yönlendir
redirectIfNotLoggedIn('login.php');

// Admin veya garson yetkisi kontrolü
if (!isAdmin() && !isWaiter()) {
    $_SESSION['error_message'] = 'Bu sayfaya erişim yetkiniz bulunmamaktadır.';
    redirect('index.php');
}

// Sayfa başlığı
$pageTitle = __('admin.waiter_calls', [], $adminLanguage);

// Ayarları al
$settings = getSetting();

// Çağrı işlemleri
$action = isset($_GET['action']) ? $_GET['action'] : 'list';
$callId = isset($_GET['id']) ? intval($_GET['id']) : 0;
$error = '';
$success = '';

// AJAX isteği kontrolü
$isAjax = isset($_GET['ajax']) && $_GET['ajax'] == 1;

// Çağrı tamamlama işlemi
if (isset($_GET['complete']) && $callId > 0) {
    // Çağrıyı tamamla
    $stmt = $conn->prepare("UPDATE waiter_calls SET status = 'completed', completed_at = NOW(), completed_by = ? WHERE id = ?");
    $stmt->bind_param("ii", $_SESSION['user_id'], $callId);
    
    if ($stmt->execute()) {
        $_SESSION['success_message'] = 'Çağrı başarıyla tamamlandı.';
    } else {
        $_SESSION['error_message'] = 'Çağrı tamamlanırken bir hata oluştu.';
    }
    
    redirect('waiter-calls.php');
}

// Çağrı silme işlemi
if (isset($_GET['delete']) && $callId > 0) {
    // Çağrıyı sil
    $stmt = $conn->prepare("DELETE FROM waiter_calls WHERE id = ?");
    $stmt->bind_param("i", $callId);
    
    if ($stmt->execute()) {
        $_SESSION['success_message'] = 'Çağrı başarıyla silindi.';
    } else {
        $_SESSION['error_message'] = 'Çağrı silinirken bir hata oluştu.';
    }
    
    redirect('waiter-calls.php');
}

// Otomatik olarak eski tamamlanmış çağrıları sil (24 saatten eski)
$conn->query("DELETE FROM waiter_calls WHERE status = 'completed' AND completed_at < DATE_SUB(NOW(), INTERVAL 24 HOUR)");

// Tüm çağrıları al
$calls = [];
$pendingCalls = 0;

// Bekleyen çağrıları al
$result = $conn->query("SELECT wc.*, t.name as table_name, t.location as table_location 
                        FROM waiter_calls wc 
                        LEFT JOIN tables t ON wc.table_id = t.id 
                        WHERE wc.status = 'pending' 
                        ORDER BY wc.created_at ASC");

if ($result) {
    while ($row = $result->fetch_assoc()) {
        $calls['pending'][] = $row;
        $pendingCalls++;
    }
}

// Tamamlanan çağrıları al (son 24 saat)
$result = $conn->query("SELECT wc.*, t.name as table_name, t.location as table_location, u.name as completed_by_name 
                        FROM waiter_calls wc 
                        LEFT JOIN tables t ON wc.table_id = t.id 
                        LEFT JOIN users u ON wc.completed_by = u.id 
                        WHERE wc.status = 'completed' AND wc.completed_at > DATE_SUB(NOW(), INTERVAL 24 HOUR) 
                        ORDER BY wc.completed_at DESC");

if ($result) {
    while ($row = $result->fetch_assoc()) {
        $calls['completed'][] = $row;
    }
}

// AJAX isteği ise JSON döndür
if ($isAjax) {
    header('Content-Type: application/json');
    echo json_encode(['newCalls' => $pendingCalls > 0]);
    exit;
}

// Aktif kullanıcı bilgilerini al
$user = getUser($_SESSION['user_id']);
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?> | <?php echo $settings['site_title'] ?? 'Dijital Menü'; ?></title>
    
    <!-- Favicon -->
    <?php if (!empty($settings['favicon'])): ?>
    <link rel="shortcut icon" href="../assets/img/<?php echo $settings['favicon']; ?>" type="image/x-icon">
    <?php endif; ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    
    <!-- Ana CSS -->
    <link href="../assets/css/style.css" rel="stylesheet">
    
    <!-- Özel Renkler -->
    <style>
        :root {
            --primary-color: <?php echo $settings['primary_color'] ?? '#3490dc'; ?>;
            --secondary-color: <?php echo $settings['secondary_color'] ?? '#38c172'; ?>;
            --text-color: <?php echo $settings['text_color'] ?? '#333333'; ?>;
            --background-color: <?php echo $settings['background_color'] ?? '#ffffff'; ?>;
        }
        
        .call-card {
            border-left: 5px solid var(--primary-color);
            transition: all 0.3s ease;
        }
        
        .call-card:hover {
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
            transform: translateY(-2px);
        }
        
        .call-card.pending {
            border-left-color: var(--danger-color);
        }
        
        .call-card.completed {
            border-left-color: var(--success-color);
        }
        
        .call-time {
            font-size: 0.9rem;
            color: #6c757d;
        }
        
        .call-table {
            font-weight: 600;
        }
        
        .call-location {
            font-size: 0.85rem;
            color: #6c757d;
        }
        
        .call-status {
            position: absolute;
            top: 10px;
            right: 10px;
        }
        
        @keyframes pulse {
            0% {
                transform: scale(1);
            }
            50% {
                transform: scale(1.05);
            }
            100% {
                transform: scale(1);
            }
        }
        
        .new-call {
            animation: pulse 1.5s infinite;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include 'includes/sidebar.php'; ?>
            
            <!-- Ana İçerik -->
            <div class="col-md-9 col-lg-10 admin-content">
                <!-- Üst Başlık -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h1 class="h3"><?php echo $pageTitle; ?></h1>
                    <div>
                        <button type="button" class="btn btn-outline-primary" onclick="location.reload();">
                            <i class="fas fa-sync-alt"></i> <?php echo __('admin.refresh', [], $adminLanguage); ?>
                        </button>
                    </div>
                </div>
                
                <!-- Oturum Mesajı -->
                <?php if (isset($_SESSION['error_message'])): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?php echo $_SESSION['error_message']; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['error_message']); ?>
                <?php endif; ?>
                
                <?php if (isset($_SESSION['success_message'])): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <?php echo $_SESSION['success_message']; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php unset($_SESSION['success_message']); ?>
                <?php endif; ?>
                
                <?php if (!empty($error)): ?>
                <div class="alert alert-danger"><?php echo $error; ?></div>
                <?php endif; ?>
                
                <?php if (!empty($success)): ?>
                <div class="alert alert-success"><?php echo $success; ?></div>
                <?php endif; ?>
                
                <!-- İçerik -->
                <div class="row">
                    <div class="col-md-6">
                        <div class="admin-card">
                            <h5 class="mb-3">
                                <i class="fas fa-bell text-danger"></i> <?php echo __('admin.pending', [], $adminLanguage); ?>
                                <?php if ($pendingCalls > 0): ?>
                                <span class="badge bg-danger ms-2"><?php echo $pendingCalls; ?></span>
                                <?php endif; ?>
                            </h5>
                            
                            <div id="waiter-calls-table">
                                <?php if (empty($calls['pending'])): ?>
                                <div class="text-center py-4">
                                    <i class="fas fa-check-circle text-success fa-3x mb-3"></i>
                                    <p class="mb-0"><?php echo __('admin.no_pending_calls', [], $adminLanguage); ?></p>
                                </div>
                                <?php else: ?>
                                <?php foreach ($calls['pending'] as $call): ?>
                                <div class="card mb-3 call-card pending <?php echo (time() - strtotime($call['created_at']) < 60) ? 'new-call' : ''; ?>">
                                    <div class="card-body position-relative">
                                        <span class="call-status badge bg-danger"><?php echo __('admin.pending', [], $adminLanguage); ?></span>
                                        <div class="call-time">
                                            <i class="far fa-clock"></i> <?php echo timeAgo($call['created_at']); ?>
                                            <br>
                                            <small><?php echo date('d.m.Y H:i:s', strtotime($call['created_at'])); ?></small>
                                        </div>
                                        <div class="call-table">
                                            <i class="fas fa-chair"></i> <?php echo $call['table_name']; ?>
                                        </div>
                                        <?php if (!empty($call['table_location'])): ?>
                                        <div class="call-location">
                                            <i class="fas fa-map-marker-alt"></i> <?php echo $call['table_location']; ?>
                                        </div>
                                        <?php endif; ?>
                                        <div class="call-type mt-2">
                                            <?php if (isset($call['call_type']) && $call['call_type'] === 'bill'): ?>
                                            <span class="badge bg-success"><i class="fas fa-file-invoice"></i> Hesap İsteniyor</span>
                                            <?php else: ?>
                                            <span class="badge bg-primary"><i class="fas fa-bell"></i> Garson Çağırılıyor</span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="mt-3">
                                            <a href="waiter-calls.php?complete=1&id=<?php echo $call['id']; ?>" class="btn btn-sm btn-success">
                                                <i class="fas fa-check"></i> <?php echo __('admin.complete', [], $adminLanguage); ?>
                                            </a>
                                            <a href="waiter-calls.php?delete=1&id=<?php echo $call['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('<?php echo __('admin.confirm_delete', [], $adminLanguage); ?>');">
                                                <i class="fas fa-trash"></i> <?php echo __('admin.delete', [], $adminLanguage); ?>
                                            </a>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="admin-card">
                            <h5 class="mb-3">
                                <i class="fas fa-check-circle text-success"></i> <?php echo __('admin.completed', [], $adminLanguage); ?>
                                <?php if (!empty($calls['completed'])): ?>
                                <span class="badge bg-secondary ms-2"><?php echo count($calls['completed']); ?></span>
                                <?php endif; ?>
                            </h5>
                            
                            <?php if (empty($calls['completed'])): ?>
                            <div class="text-center py-4">
                                <i class="fas fa-history text-muted fa-3x mb-3"></i>
                                <p class="mb-0"><?php echo __('admin.no_completed_calls', [], $adminLanguage); ?></p>
                            </div>
                            <?php else: ?>
                            <?php foreach ($calls['completed'] as $call): ?>
                            <div class="card mb-3 call-card completed">
                                <div class="card-body position-relative">
                                    <span class="call-status badge bg-success"><?php echo __('admin.completed', [], $adminLanguage); ?></span>
                                    <div class="call-time">
                                        <i class="far fa-clock"></i> <?php echo timeAgo($call['completed_at']); ?> <?php echo __('admin.completed_at', [], $adminLanguage); ?>
                                        <br>
                                        <small><?php echo date('d.m.Y H:i:s', strtotime($call['completed_at'])); ?></small>
                                    </div>
                                    <div class="call-table">
                                        <i class="fas fa-chair"></i> <?php echo $call['table_name']; ?>
                                    </div>
                                    <?php if (!empty($call['table_location'])): ?>
                                    <div class="call-location">
                                        <i class="fas fa-map-marker-alt"></i> <?php echo $call['table_location']; ?>
                                    </div>
                                    <?php endif; ?>
                                    <div class="call-type mt-2">
                                        <?php if (isset($call['call_type']) && $call['call_type'] === 'bill'): ?>
                                        <span class="badge bg-success"><i class="fas fa-file-invoice"></i> Hesap İsteniyor</span>
                                        <?php else: ?>
                                        <span class="badge bg-primary"><i class="fas fa-bell"></i> Garson Çağırılıyor</span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="mt-2 text-muted">
                                        <small>
                                            <i class="fas fa-user"></i> <?php echo $call['completed_by_name']; ?> <?php echo __('admin.completed_by', [], $adminLanguage); ?>
                                        </small>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <!-- Ana JS -->
    <script src="../assets/js/script.js"></script>
    
    <!-- Bildirim Sesi -->
    <audio id="notification-sound" src="../assets/sounds/notification.mp3" preload="auto"></audio>
    
    <!-- Yeni Çağrı Kontrolü ve Bildirim -->
    <script>
        // Bildirim sesi çalma fonksiyonu
        function playNotificationSound() {
            const sound = document.getElementById('notification-sound');
            if (sound) {
                // Sesi çal
                sound.play().catch(e => {
                    console.log('Ses çalma hatası:', e);
                });
            }
        }
        
        // Tarayıcı bildirimi gösterme fonksiyonu
        function showNotification(title, options) {
            // Bildirim izni kontrolü
            if (!("Notification" in window)) {
                console.log("Bu tarayıcı bildirim desteği sunmuyor");
                return;
            }
            
            // İzin durumuna göre işlem yap
            if (Notification.permission === "granted") {
                // Bildirim göster
                const notification = new Notification(title, options);
                
                // Bildirime tıklandığında sayfayı yenile
                notification.onclick = function() {
                    window.focus();
                    location.reload();
                };
            } else if (Notification.permission !== "denied") {
                // İzin iste
                Notification.requestPermission().then(function(permission) {
                    if (permission === "granted") {
                        const notification = new Notification(title, options);
                        
                        notification.onclick = function() {
                            window.focus();
                            location.reload();
                        };
                    }
                });
            }
        }
        
        // Sayfa yüklendiğinde yeni çağrı varsa ses çal ve bildirim göster
        <?php if (!empty($calls['pending']) && time() - strtotime($calls['pending'][0]['created_at']) < 60): ?>
        document.addEventListener('DOMContentLoaded', function() {
            // Ses çal
            playNotificationSound();
            
            // Bildirim başlığı ve içeriği
            <?php if (isset($calls['pending'][0]['call_type']) && $calls['pending'][0]['call_type'] === 'bill'): ?>
            // Hesap isteği bildirimi
            showNotification('Yeni Hesap İsteği', {
                body: '<?php echo $calls['pending'][0]['table_name']; ?> masası hesap istiyor!',
                icon: '../assets/img/<?php echo $settings['favicon'] ?? 'favicon.ico'; ?>',
                tag: 'waiter-call-<?php echo $calls['pending'][0]['id']; ?>'
            });
            <?php else: ?>
            // Garson çağrısı bildirimi
            showNotification('Yeni Garson Çağrısı', {
                body: '<?php echo $calls['pending'][0]['table_name']; ?> masasından yeni bir çağrı var!',
                icon: '../assets/img/<?php echo $settings['favicon'] ?? 'favicon.ico'; ?>',
                tag: 'waiter-call-<?php echo $calls['pending'][0]['id']; ?>'
            });
            <?php endif; ?>
        });
        <?php endif; ?>
        
        // Periyodik olarak yeni çağrıları kontrol et (30 saniyede bir)
        setInterval(function() {
            fetch('waiter-calls.php?ajax=1')
                .then(response => response.json())
                .then(data => {
                    if (data.newCalls) {
                        // Yeni çağrı varsa sayfayı yenile
                        location.reload();
                    }
                })
                .catch(error => console.error('Çağrı kontrolü hatası:', error));
        }, 30000);
        
        // Bildirim izni iste
        document.addEventListener('DOMContentLoaded', function() {
            if ("Notification" in window && Notification.permission !== "granted" && Notification.permission !== "denied") {
                // Kullanıcıya bildirim izni iste
                Notification.requestPermission();
            }
        });
    </script>
</body>
</html>
