<?php
/**
 * Dijital Menü - Kimlik Doğrulama Fonksiyonları
 */

// Gerekli dosyaları dahil et
require_once __DIR__ . '/functions.php';

/**
 * Kullanıcı girişi
 * 
 * @param string $username Kullanıcı adı
 * @param string $password Şifre
 * @return array|bool Başarılı ise kullanıcı bilgileri, başarısız ise false
 */
function login($username, $password) {
    global $conn;
    
    $stmt = $conn->prepare("SELECT * FROM users WHERE username = ? AND status = 1");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        $user = $result->fetch_assoc();
        
        if (verifyPassword($password, $user['password'])) {
            // Oturum bilgilerini kaydet
            $_SESSION['logged_in'] = true;
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['user_role'] = $user['role'];
            
            // Son giriş zamanını güncelle
            $stmt = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
            $stmt->bind_param("i", $user['id']);
            $stmt->execute();
            
            return $user;
        }
    }
    
    return false;
}

/**
 * Kullanıcı çıkışı
 * 
 * @return void
 */
function logout() {
    // Oturum bilgilerini temizle
    $_SESSION = [];
    
    // Oturum çerezini sil
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    
    // Oturumu sonlandır
    session_destroy();
}

/**
 * Kullanıcı bilgilerini al
 * 
 * @param int $userId Kullanıcı ID
 * @return array|bool Başarılı ise kullanıcı bilgileri, başarısız ise false
 */
function getUser($userId) {
    global $conn;
    
    $stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        return $result->fetch_assoc();
    }
    
    return false;
}

/**
 * Giriş yapılmışsa yönlendir
 * 
 * @param string $url Yönlendirilecek URL
 * @return void
 */
function redirectIfLoggedIn($url) {
    if (isLoggedIn()) {
        // Kullanıcı rolüne göre yönlendirme
        if ($_SESSION['user_role'] === 'waiter') {
            redirect('waiter-calls.php');
        } else {
            redirect($url);
        }
    }
}

/**
 * Giriş yapılmamışsa yönlendir
 * 
 * @param string $url Yönlendirilecek URL
 * @return void
 */
function redirectIfNotLoggedIn($url) {
    if (!isLoggedIn()) {
        redirect($url);
    }
}

/**
 * Yetki kontrolü
 * 
 * @param string $role Gerekli rol
 * @param string $redirectUrl Yönlendirilecek URL
 * @return void
 */
function requirePermission($role, $redirectUrl) {
    if (!isLoggedIn()) {
        redirect($redirectUrl);
    }
    
    if ($role === 'admin' && !isAdmin()) {
        $_SESSION['error_message'] = 'Bu sayfaya erişim yetkiniz bulunmamaktadır.';
        redirect($redirectUrl);
    }
    
    if ($role === 'manager' && !isManager()) {
        $_SESSION['error_message'] = 'Bu sayfaya erişim yetkiniz bulunmamaktadır.';
        redirect($redirectUrl);
    }
}

/**
 * Hatırla beni kontrolü
 * 
 * @return void
 */
function checkRememberMe() {
    global $conn;
    
    if (!isLoggedIn() && isset($_COOKIE['remember_token']) && !empty($_COOKIE['remember_token'])) {
        $token = $_COOKIE['remember_token'];
        
        $stmt = $conn->prepare("SELECT * FROM users WHERE remember_token = ? AND remember_expires > NOW() AND status = 1");
        $stmt->bind_param("s", $token);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 1) {
            $user = $result->fetch_assoc();
            
            // Oturum bilgilerini kaydet
            $_SESSION['logged_in'] = true;
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['user_role'] = $user['role'];
            
            // Son giriş zamanını güncelle
            $stmt = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
            $stmt->bind_param("i", $user['id']);
            $stmt->execute();
            
            // Token süresini uzat
            $expires = time() + (86400 * 30); // 30 gün
            setcookie('remember_token', $token, $expires, '/');
            
            $stmt = $conn->prepare("UPDATE users SET remember_expires = ? WHERE id = ?");
            $expiresDate = date('Y-m-d H:i:s', $expires);
            $stmt->bind_param("si", $expiresDate, $user['id']);
            $stmt->execute();
        }
    }
}

// Hatırla beni kontrolü
checkRememberMe();
