<?php
/**
 * Geri Bildirim Fonksiyonları
 */

/**
 * Yeni bir geri bildirim ekler
 * 
 * @param string $name Gönderen adı
 * @param string $email Gönderen e-posta adresi
 * @param string $message Geri bildirim mesajı
 * @return bool İşlem başarılı ise true, değilse false döner
 */
function addFeedback($name, $email, $message) {
    global $conn;
    
    // IP adresini al
    $ipAddress = $_SERVER['REMOTE_ADDR'];
    
    // XSS koruması
    $name = htmlspecialchars($name);
    $email = htmlspecialchars($email);
    $message = htmlspecialchars($message);
    
    // Boş isim kontrolü
    if (empty($name)) {
        $name = 'Misafir';
    }
    
    // SQL sorgusu
    $stmt = $conn->prepare("INSERT INTO feedback (name, email, message, ip_address) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("ssss", $name, $email, $message, $ipAddress);
    
    return $stmt->execute();
}

/**
 * Tüm geri bildirimleri getirir
 * 
 * @param int $status Geri bildirim durumu (null: tümü, 0: beklemede, 1: okundu, 2: yanıtlandı, 3: arşivlendi)
 * @param int $limit Maksimum kayıt sayısı
 * @param int $offset Başlangıç kaydı
 * @param string $search Arama filtresi
 * @param string $startDate Başlangıç tarihi
 * @param string $endDate Bitiş tarihi
 * @return array Geri bildirimler dizisi
 */
function getFeedbacks($status = null, $limit = 100, $offset = 0, $search = null, $startDate = null, $endDate = null) {
    global $conn;
    
    // SQL sorgusu
    $sql = "SELECT * FROM feedback";
    
    $conditions = [];
    $params = [];
    $types = "";
    
    // Durum filtresi
    if ($status !== null) {
        $conditions[] = "status = ?";
        $params[] = intval($status);
        $types .= "i";
    }
    
    // Arama filtresi
    if ($search !== null && !empty($search)) {
        $conditions[] = "(name LIKE ? OR email LIKE ? OR message LIKE ?)";
        $searchTerm = "%" . $search . "%";
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $types .= "sss";
    }
    
    // Tarih filtreleri
    if ($startDate !== null && !empty($startDate)) {
        $conditions[] = "DATE(created_at) >= ?";
        $params[] = $startDate;
        $types .= "s";
    }
    
    if ($endDate !== null && !empty($endDate)) {
        $conditions[] = "DATE(created_at) <= ?";
        $params[] = $endDate;
        $types .= "s";
    }
    
    // WHERE koşullarını ekle
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    // Sıralama ve limit
    $sql .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
    $params[] = intval($limit);
    $params[] = intval($offset);
    $types .= "ii";
    
    $stmt = $conn->prepare($sql);
    
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $feedbacks = [];
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $feedbacks[] = $row;
        }
    }
    
    return $feedbacks;
}

/**
 * Belirli bir geri bildirimi getirir
 * 
 * @param int $id Geri bildirim ID
 * @return array|null Geri bildirim bilgileri veya null
 */
function getFeedback($id) {
    global $conn;
    
    $stmt = $conn->prepare("SELECT * FROM feedback WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        return $result->fetch_assoc();
    }
    
    return null;
}

/**
 * Geri bildirim durumunu günceller
 * 
 * @param int $id Geri bildirim ID
 * @param int $status Yeni durum (0: beklemede, 1: okundu, 2: yanıtlandı, 3: arşivlendi)
 * @return bool İşlem başarılı ise true, değilse false döner
 */
function updateFeedbackStatus($id, $status) {
    global $conn;
    
    $stmt = $conn->prepare("UPDATE feedback SET status = ? WHERE id = ?");
    $stmt->bind_param("ii", $status, $id);
    
    return $stmt->execute();
}

/**
 * Geri bildirim siler
 * 
 * @param int $id Geri bildirim ID
 * @return bool İşlem başarılı ise true, değilse false döner
 */
function deleteFeedback($id) {
    global $conn;
    
    $stmt = $conn->prepare("DELETE FROM feedback WHERE id = ?");
    $stmt->bind_param("i", $id);
    
    return $stmt->execute();
}

/**
 * Toplam geri bildirim sayısını getirir
 * 
 * @param int $status Geri bildirim durumu (null: tümü, 0: beklemede, 1: okundu, 2: yanıtlandı, 3: arşivlendi)
 * @return int Geri bildirim sayısı
 */
function getFeedbackCount($status = null) {
    global $conn;
    
    // SQL sorgusu
    $sql = "SELECT COUNT(*) as count FROM feedback";
    
    // Durum filtresi
    if ($status !== null) {
        $sql .= " WHERE status = " . intval($status);
    }
    
    $result = $conn->query($sql);
    $row = $result->fetch_assoc();
    
    return $row['count'];
}

/**
 * Geri bildirim durumunu metin olarak döndürür
 * 
 * @param int $status Durum kodu
 * @return string Durum metni
 */
function getFeedbackStatusText($status) {
    switch ($status) {
        case 0:
            return 'Beklemede';
        case 1:
            return 'Okundu';
        case 2:
            return 'Yanıtlandı';
        case 3:
            return 'Arşivlendi';
        default:
            return 'Bilinmiyor';
    }
}

/**
 * Geri bildirim durumuna göre renk sınıfı döndürür
 * 
 * @param int $status Durum kodu
 * @return string Bootstrap renk sınıfı
 */
function getFeedbackStatusClass($status) {
    switch ($status) {
        case 0:
            return 'warning'; // Beklemede: Sarı
        case 1:
            return 'info';    // Okundu: Mavi
        case 2:
            return 'success'; // Yanıtlandı: Yeşil
        case 3:
            return 'secondary'; // Arşivlendi: Gri
        default:
            return 'light';
    }
}
