<?php
// Veritabanı bağlantısını dahil et
require_once __DIR__ . '/../config/database.php';

/**
 * Metni temizle (XSS koruması)
 * 
 * @param string $data Temizlenecek veri
 * @return string Temizlenmiş veri
 */
function clean($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

/**
 * Yönlendirme
 * 
 * @param string $url Yönlendirilecek URL
 * @return void
 */
function redirect($url) {
    header("Location: $url");
    exit;
}

/**
 * Şifre hashleme
 * 
 * @param string $password Şifre
 * @return string Hashlenmiş şifre
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Şifre doğrulama
 * 
 * @param string $password Şifre
 * @param string $hash Hashlenmiş şifre
 * @return bool Doğrulama sonucu
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Giriş yapılmış mı kontrolü
 * 
 * @return bool Giriş durumu
 */
function isLoggedIn() {
    return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
}

/**
 * Admin mi kontrolü
 * 
 * @return bool Admin durumu
 */
function isAdmin() {
    return isLoggedIn() && $_SESSION['user_role'] === 'admin';
}

/**
 * Yönetici mi kontrolü
 * 
 * @return bool Yönetici durumu
 */
function isManager() {
    return isLoggedIn() && ($_SESSION['user_role'] === 'admin' || $_SESSION['user_role'] === 'manager');
}

/**
 * Garson yetkisi kontrolü
 * 
 * @return bool
 */
function isWaiter() {
    return isLoggedIn() && ($_SESSION['user_role'] === 'waiter' || $_SESSION['user_role'] === 'admin' || $_SESSION['user_role'] === 'manager');
}

/**
 * Ayarları al
 * 
 * @param string $key Ayar anahtarı (null ise tüm ayarlar)
 * @return array|string|null Ayar değeri
 */
function getSetting($key = null) {
    global $conn;
    
    $settings = [];
    $result = $conn->query("SELECT setting_key, setting_value FROM settings");
    
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
    
    if ($key !== null) {
        return $settings[$key] ?? null;
    }
    
    return $settings;
}

/**
 * Kategorileri al
 * 
 * @param bool $activeOnly Sadece aktif kategoriler mi
 * @return array Kategoriler
 */
function getCategories($activeOnly = false) {
    global $conn;
    
    $sql = "SELECT * FROM categories";
    
    if ($activeOnly) {
        $sql .= " WHERE status = 1";
    }
    
    $sql .= " ORDER BY sort_order ASC, name ASC";
    
    $result = $conn->query($sql);
    $categories = [];
    
    while ($row = $result->fetch_assoc()) {
        $categories[] = $row;
    }
    
    return $categories;
}

/**
 * Menü öğelerini al
 * 
 * @param int|null $categoryId Kategori ID (null ise tüm öğeler)
 * @param bool $activeOnly Sadece aktif öğeler mi
 * @return array Menü öğeleri
 */
function getItems($categoryId = null, $activeOnly = false) {
    global $conn;
    
    $sql = "SELECT * FROM items";
    $conditions = [];
    
    if ($categoryId !== null) {
        $conditions[] = "category_id = $categoryId";
    }
    
    if ($activeOnly) {
        $conditions[] = "status = 1";
    }
    
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    $sql .= " ORDER BY sort_order ASC, name ASC";
    
    $result = $conn->query($sql);
    $items = [];
    
    while ($row = $result->fetch_assoc()) {
        $items[] = $row;
    }
    
    return $items;
}

/**
 * Alerjenleri al
 * 
 * @return array Alerjenler
 */
function getAllergens() {
    global $conn;
    
    $sql = "SELECT * FROM allergens ORDER BY name ASC";
    $result = $conn->query($sql);
    $allergens = [];
    
    while ($row = $result->fetch_assoc()) {
        $allergens[] = $row;
    }
    
    return $allergens;
}

/**
 * Menü öğesinin alerjenlerini al
 * 
 * @param int $itemId Menü öğesi ID
 * @return array Alerjenler
 */
function getItemAllergens($itemId) {
    global $conn;
    
    $sql = "SELECT a.* FROM allergens a 
            JOIN item_allergens ia ON a.id = ia.allergen_id 
            WHERE ia.item_id = $itemId 
            ORDER BY a.name ASC";
    
    $result = $conn->query($sql);
    $allergens = [];
    
    while ($row = $result->fetch_assoc()) {
        $allergens[] = $row;
    }
    
    return $allergens;
}

/**
 * Masaları al
 * 
 * @param string|null $status Durum filtresi (null ise tüm masalar)
 * @return array Masalar
 */
function getTables($status = null) {
    global $conn;
    
    $sql = "SELECT * FROM tables";
    
    if ($status !== null) {
        $sql .= " WHERE status = '" . clean($status) . "'";
    }
    
    $sql .= " ORDER BY name ASC";
    
    $result = $conn->query($sql);
    $tables = [];
    
    while ($row = $result->fetch_assoc()) {
        $tables[] = $row;
    }
    
    return $tables;
}

/**
 * Garson çağrılarını al
 * 
 * @param string|null $status Durum filtresi (null ise tüm çağrılar)
 * @return array Garson çağrıları
 */
function getWaiterCalls($status = null) {
    global $conn;
    
    $sql = "SELECT wc.*, t.name as table_name FROM waiter_calls wc 
            JOIN tables t ON wc.table_id = t.id";
    
    if ($status !== null) {
        $sql .= " WHERE wc.status = '" . clean($status) . "'";
    }
    
    $sql .= " ORDER BY wc.created_at DESC";
    
    $result = $conn->query($sql);
    $calls = [];
    
    while ($row = $result->fetch_assoc()) {
        $calls[] = $row;
    }
    
    return $calls;
}

/**
 * Garson çağrısı oluştur
 * 
 * @param int $tableId Masa ID
 * @param string $callType Çağrı türü (waiter: Garson Çağır, bill: Hesap İste)
 * @return bool İşlem sonucu
 */
function callWaiter($tableId, $callType = 'waiter') {
    global $conn;
    
    $stmt = $conn->prepare("INSERT INTO waiter_calls (table_id, status, call_type) VALUES (?, 'pending', ?)");
    $stmt->bind_param("is", $tableId, $callType);
    
    return $stmt->execute();
}

/**
 * Garson çağrısını tamamla
 * 
 * @param int $callId Çağrı ID
 * @return bool İşlem sonucu
 */
function completeWaiterCall($callId) {
    global $conn;
    
    $stmt = $conn->prepare("UPDATE waiter_calls SET status = 'completed' WHERE id = ?");
    $stmt->bind_param("i", $callId);
    
    return $stmt->execute();
}

/**
 * Dilleri al
 * 
 * @param bool $activeOnly Sadece aktif diller mi
 * @return array Diller
 */
function getLanguages($activeOnly = true) {
    global $conn;
    
    $sql = "SELECT * FROM languages";
    
    if ($activeOnly) {
        $sql .= " WHERE status = 1";
    }
    
    $sql .= " ORDER BY is_default DESC, name ASC";
    
    $result = $conn->query($sql);
    $languages = [];
    
    while ($row = $result->fetch_assoc()) {
        $languages[] = $row;
    }
    
    return $languages;
}

/**
 * Dil ayarını belirle
 * 
 * @param string $language Dil kodu
 * @return void
 */
function setLanguage($language) {
    global $conn;
    
    // Dil geçerli mi kontrol et
    $stmt = $conn->prepare("SELECT * FROM languages WHERE code = ? AND status = 1");
    $stmt->bind_param("s", $language);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        // Geçerli değilse varsayılan dili al
        $result = $conn->query("SELECT code FROM languages WHERE is_default = 1 AND status = 1");
        if ($result->num_rows > 0) {
            $language = $result->fetch_assoc()['code'];
        } else {
            $language = 'tr'; // Varsayılan dil
        }
    }
    
    // Dil ayarını oturuma kaydet
    $_SESSION['language'] = $language;
}

/**
 * Çeviri al
 * 
 * @param string $key Çeviri anahtarı
 * @param string $language Dil kodu
 * @param string $type Çeviri türü (setting, category, item, allergen)
 * @param int $itemId Öğe ID (kategori, ürün veya alerjen için)
 * @param string $field Alan adı (name, description vb.)
 * @return string|null Çeviri
 */
function translate($key, $language, $type = 'setting', $itemId = 0, $field = '') {
    global $conn;
    
    // Dil ID'sini al
    $stmt = $conn->prepare("SELECT id FROM languages WHERE code = ?");
    $stmt->bind_param("s", $language);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        return null;
    }
    
    $languageId = $result->fetch_assoc()['id'];
    
    // Çeviri türüne göre sorgu yap
    if ($type === 'setting') {
        // Genel metin çevirisi
        $stmt = $conn->prepare("SELECT translation FROM translations WHERE language_id = ? AND item_type = 'setting' AND field_name = ?");
        $stmt->bind_param("is", $languageId, $key);
    } else {
        // Kategori, ürün veya alerjen çevirisi
        $stmt = $conn->prepare("SELECT translation FROM translations WHERE language_id = ? AND item_type = ? AND item_id = ? AND field_name = ?");
        $stmt->bind_param("isis", $languageId, $type, $itemId, $field);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        return $result->fetch_assoc()['translation'];
    }
    
    return null;
}

/**
 * Para formatı
 * 
 * @param float $amount Miktar
 * @param string $currencySymbol Para birimi sembolü
 * @param string $position Sembol pozisyonu (before/after)
 * @return string Formatlanmış miktar
 */
function formatMoney($amount, $currencySymbol = null, $position = null) {
    // Ayarlardan para birimi sembolü ve pozisyonu al
    if ($currencySymbol === null) {
        $currencySymbol = getSetting('currency_symbol') ?? '₺';
    }
    
    if ($position === null) {
        $position = getSetting('currency_position') ?? 'after';
    }
    
    // Formatla
    $formattedAmount = number_format($amount, 2, ',', '.');
    
    if ($position === 'before') {
        return $currencySymbol . $formattedAmount;
    } else {
        return $formattedAmount . ' ' . $currencySymbol;
    }
}

/**
 * Dosya yükleme
 * 
 * @param array $file $_FILES dizisi
 * @param string $uploadDir Yükleme dizini
 * @param array $allowedTypes İzin verilen dosya türleri (boş ise tüm türler)
 * @param int $maxSize Maksimum dosya boyutu (byte)
 * @return string|bool Başarılı ise dosya adı, başarısız ise false
 */
function uploadFile($file, $uploadDir, $allowedTypes = [], $maxSize = 5242880) {
    // Hata kontrolü
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return false;
    }
    
    // Boyut kontrolü
    if ($file['size'] > $maxSize) {
        return false;
    }
    
    // Tür kontrolü
    if (!empty($allowedTypes)) {
        $fileType = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($fileType, $allowedTypes)) {
            return false;
        }
    }
    
    // Dizin kontrolü
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Benzersiz dosya adı oluştur
    $fileName = uniqid() . '.' . strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $filePath = $uploadDir . '/' . $fileName;
    
    // Dosyayı yükle
    if (move_uploaded_file($file['tmp_name'], $filePath)) {
        return $fileName;
    }
    
    return false;
}

/**
 * QR kod oluştur
 * 
 * @param int $tableId Masa ID
 * @param string $fileName Dosya adı
 * @param string $uploadDir Yükleme dizini
 * @return string|bool Başarılı ise dosya adı, başarısız ise false
 */
function generateQRCode($tableId, $fileName, $uploadDir) {
    // Dizin kontrolü ve oluşturma
    if (!file_exists($uploadDir)) {
        if (!mkdir($uploadDir, 0755, true)) {
            // Dizin oluşturulamadı, hata mesajı
            error_log("QR kod dizini oluşturulamadı: $uploadDir");
            return false;
        }
    }
    
    // Masa bilgisini Base64 ile kodla
    $encodedTableId = base64_encode("Table:$tableId");
    
    // Site URL'sini ayarlardan al
    $settings = getSetting();
    $siteUrl = $settings['site_url'] ?? 'http://localhost/digital-menu';
    
    // QR kod verisi oluştur
    $qrData = "$siteUrl/index.php?qr=$encodedTableId";
    
    // QR kod kütüphanesi olmadığı için Google Charts API kullanıyoruz
    $url = 'https://chart.googleapis.com/chart?chs=300x300&cht=qr&chl=' . urlencode($qrData) . '&choe=UTF-8';
    
    // Resmi indir
    $content = @file_get_contents($url);
    
    if ($content === false) {
        // API'ye erişim hatası, farklı bir API deneyelim
        $url = 'https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=' . urlencode($qrData);
        
        // Resmi indir
        $content = @file_get_contents($url);
        
        if ($content === false) {
            // İkinci API'ye de erişim hatası, hata mesajı
            error_log("QR kod oluşturulamadı: API'lere erişim hatası");
            return false;
        }
        
        // Dosyayı kaydet
        $filePath = $uploadDir . '/' . $fileName;
        
        if (file_put_contents($filePath, $content) === false) {
            error_log("QR kod dosyası kaydedilemedi: $filePath");
            return false;
        }
        
        return $fileName;
    }
    
    // Dosyayı kaydet
    $filePath = $uploadDir . '/' . $fileName;
    
    if (file_put_contents($filePath, $content) === false) {
        error_log("QR kod dosyası kaydedilemedi: $filePath");
        return false;
    }
    
    return $fileName;
}

/**
 * Oturum mesajı oluştur
 * 
 * @param string $message Mesaj
 * @param string $type Mesaj türü (success/error/info/warning)
 * @return void
 */
function setMessage($message, $type = 'success') {
    $_SESSION[$type . '_message'] = $message;
}

/**
 * Oturum mesajını al
 * 
 * @param string $type Mesaj türü (success/error/info/warning)
 * @return string|null Mesaj
 */
function getMessage($type = 'success') {
    if (isset($_SESSION[$type . '_message'])) {
        $message = $_SESSION[$type . '_message'];
        unset($_SESSION[$type . '_message']);
        return $message;
    }
    
    return null;
}

/**
 * Slug oluştur
 * 
 * @param string $text Metin
 * @return string Slug
 */
function createSlug($text) {
    // Türkçe karakterleri değiştir
    $text = str_replace(
        ['ı', 'ğ', 'ü', 'ş', 'ö', 'ç', 'İ', 'Ğ', 'Ü', 'Ş', 'Ö', 'Ç'],
        ['i', 'g', 'u', 's', 'o', 'c', 'I', 'G', 'U', 'S', 'O', 'C'],
        $text
    );
    
    // Slug oluştur
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9\s-]/', '', $text);
    $text = preg_replace('/[\s-]+/', '-', $text);
    $text = trim($text, '-');
    
    return $text;
}

/**
 * Tarih formatı
 * 
 * @param string $date Tarih
 * @param string $format Format
 * @return string Formatlanmış tarih
 */
function formatDate($date, $format = 'd.m.Y H:i') {
    return date($format, strtotime($date));
}

/**
 * Zaman farkı
 * 
 * @param string $date Tarih
 * @return string Zaman farkı metni
 */
function timeAgo($date) {
    $timestamp = strtotime($date);
    $difference = time() - $timestamp;
    
    if ($difference < 60) {
        return 'Az önce';
    } elseif ($difference < 3600) {
        return floor($difference / 60) . ' dakika önce';
    } elseif ($difference < 86400) {
        return floor($difference / 3600) . ' saat önce';
    } elseif ($difference < 604800) {
        return floor($difference / 86400) . ' gün önce';
    } elseif ($difference < 2592000) {
        return floor($difference / 604800) . ' hafta önce';
    } elseif ($difference < 31536000) {
        return floor($difference / 2592000) . ' ay önce';
    } else {
        return floor($difference / 31536000) . ' yıl önce';
    }
}

// Dil fonksiyonlarını dahil et
require_once __DIR__ . '/language.php';

// Puanlama fonksiyonlarını dahil et
require_once __DIR__ . '/rating_functions.php';

/**
 * Admin paneli çeviri anahtarları
 * 
 * @return array Admin paneli çeviri anahtarları
 */
function getAdminTranslationKeys() {
    // Dil dosyasından admin paneli çevirilerini al
    $lang = loadLanguage();
    
    $adminKeys = [];
    foreach ($lang as $key => $value) {
        // Admin paneli ile ilgili anahtarları filtrele
        if (strpos($key, 'admin_') === 0 || 
            in_array($key, [
                'dashboard', 'save', 'cancel', 'edit', 'delete', 'add', 'search', 'filter',
                'status', 'active', 'inactive', 'actions', 'confirm_delete', 'yes', 'no',
                'success', 'error', 'warning', 'info', 'menu', 'categories', 'items',
                'allergens', 'tables', 'waiter_calls', 'languages', 'translations', 'users',
                'settings', 'logout', 'category_name', 'category_description', 'parent_category',
                'main_category', 'category_icon', 'sort_order', 'add_category', 'edit_category',
                'category_list', 'item_name', 'item_description', 'price', 'category',
                'select_category', 'item_image', 'special', 'popular', 'vegetarian', 'vegan',
                'gluten_free', 'add_item', 'edit_item', 'item_list', 'allergen_name',
                'allergen_icon', 'add_allergen', 'edit_allergen', 'allergen_list', 'table_name',
                'table_number', 'capacity', 'qr_code', 'add_table', 'edit_table', 'table_list',
                'generate_qr', 'download_qr', 'waiter_call', 'table', 'time', 'status',
                'pending', 'completed', 'mark_completed', 'language_code', 'language_name',
                'flag', 'default_language', 'make_default', 'add_language', 'edit_language',
                'language_list', 'select_language', 'translation_type', 'general_texts', 'key',
                'default_value', 'translation', 'save_translations', 'username', 'password',
                'confirm_password', 'email', 'name', 'role', 'admin', 'manager', 'waiter',
                'add_user', 'edit_user', 'user_list', 'site_title', 'logo', 'favicon',
                'currency', 'currency_symbol', 'currency_position', 'before', 'after',
                'primary_color', 'secondary_color', 'text_color', 'background_color',
                'save_settings'
            ])) {
            $adminKeys[$key] = $value;
        }
    }
    
    return $adminKeys;
}

/**
 * Site çeviri anahtarları
 * 
 * @return array Site çeviri anahtarları
 */
function getSiteTranslationKeys() {
    // Dil dosyasından site çevirilerini al
    $lang = loadLanguage();
    
    $siteKeys = [];
    foreach ($lang as $key => $value) {
        // Site ile ilgili anahtarları filtrele
        if (strpos($key, 'admin_') !== 0 && 
            in_array($key, [
                'menu', 'categories', 'all_categories', 'special_items', 'popular_items',
                'vegetarian', 'vegan', 'gluten_free', 'allergens', 'call_waiter', 'table',
                'search', 'search_placeholder', 'no_items_found', 'footer_text', 'call_sent',
                'call_error', 'back_to_menu', 'price', 'description', 'contains_allergens',
                'language', 'home', 'contact', 'about'
            ])) {
            $siteKeys[$key] = $value;
        }
    }
    
    return $siteKeys;
}
