<?php
/**
 * Dijital Menü - Slider Fonksiyonları
 */

/**
 * Tüm sliderları getirir
 * 
 * @param bool $activeOnly Sadece aktif sliderları getir
 * @return array Sliderlar dizisi
 */
function getSliders($activeOnly = false) {
    global $conn;
    
    $sql = "SELECT * FROM sliders";
    
    if ($activeOnly) {
        $sql .= " WHERE status = 1";
    }
    
    $sql .= " ORDER BY sort_order ASC";
    
    $result = $conn->query($sql);
    $sliders = [];
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $sliders[] = $row;
        }
    }
    
    return $sliders;
}

/**
 * Belirli bir slider'ı getirir
 * 
 * @param int $id Slider ID
 * @return array|null Slider verisi veya null
 */
function getSlider($id) {
    global $conn;
    
    $stmt = $conn->prepare("SELECT * FROM sliders WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        return $result->fetch_assoc();
    }
    
    return null;
}

/**
 * Yeni bir slider ekler
 * 
 * @param array $data Slider verileri
 * @return bool İşlem başarılı mı
 */
function addSlider($data) {
    global $conn;
    
    $title = isset($data['title']) ? clean($data['title']) : '';
    $description = isset($data['description']) ? clean($data['description']) : '';
    $image = isset($data['image']) ? clean($data['image']) : '';
    $link = isset($data['link']) ? clean($data['link']) : '';
    $sortOrder = isset($data['sort_order']) ? intval($data['sort_order']) : 0;
    $status = isset($data['status']) ? intval($data['status']) : 1;
    
    $stmt = $conn->prepare("INSERT INTO sliders (title, description, image, link, sort_order, status) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("ssssii", $title, $description, $image, $link, $sortOrder, $status);
    
    return $stmt->execute();
}

/**
 * Slider günceller
 * 
 * @param int $id Slider ID
 * @param array $data Güncellenecek veriler
 * @return bool İşlem başarılı mı
 */
function updateSlider($id, $data) {
    global $conn;
    
    $title = isset($data['title']) ? clean($data['title']) : '';
    $description = isset($data['description']) ? clean($data['description']) : '';
    $link = isset($data['link']) ? clean($data['link']) : '';
    $sortOrder = isset($data['sort_order']) ? intval($data['sort_order']) : 0;
    $status = isset($data['status']) ? intval($data['status']) : 1;
    
    // Resim varsa güncelle
    if (!empty($data['image'])) {
        $image = clean($data['image']);
        $stmt = $conn->prepare("UPDATE sliders SET title = ?, description = ?, image = ?, link = ?, sort_order = ?, status = ? WHERE id = ?");
        $stmt->bind_param("ssssiis", $title, $description, $image, $link, $sortOrder, $status, $id);
    } else {
        // Resim yoksa diğer alanları güncelle
        $stmt = $conn->prepare("UPDATE sliders SET title = ?, description = ?, link = ?, sort_order = ?, status = ? WHERE id = ?");
        $stmt->bind_param("sssiis", $title, $description, $link, $sortOrder, $status, $id);
    }
    
    return $stmt->execute();
}

/**
 * Slider siler
 * 
 * @param int $id Slider ID
 * @return bool İşlem başarılı mı
 */
function deleteSlider($id) {
    global $conn;
    
    // Önce slider'ı al
    $slider = getSlider($id);
    
    if ($slider) {
        // Resmi sil
        if (!empty($slider['image'])) {
            $imagePath = '../assets/uploads/' . $slider['image'];
            if (file_exists($imagePath)) {
                unlink($imagePath);
            }
        }
        
        // Slider'ı veritabanından sil
        $stmt = $conn->prepare("DELETE FROM sliders WHERE id = ?");
        $stmt->bind_param("i", $id);
        
        return $stmt->execute();
    }
    
    return false;
}

/**
 * Slider durumunu günceller
 * 
 * @param int $id Slider ID
 * @param int $status Yeni durum (0: pasif, 1: aktif)
 * @return bool İşlem başarılı mı
 */
function updateSliderStatus($id, $status) {
    global $conn;
    
    $stmt = $conn->prepare("UPDATE sliders SET status = ? WHERE id = ?");
    $stmt->bind_param("ii", $status, $id);
    
    return $stmt->execute();
}

/**
 * Slider sayısını getirir
 * 
 * @param bool $activeOnly Sadece aktif sliderları say
 * @return int Slider sayısı
 */
function getSliderCount($activeOnly = false) {
    global $conn;
    
    $sql = "SELECT COUNT(*) as count FROM sliders";
    
    if ($activeOnly) {
        $sql .= " WHERE status = 1";
    }
    
    $result = $conn->query($sql);
    $row = $result->fetch_assoc();
    
    return $row['count'];
}
