<?php
session_start();
require_once '../config/db.php';
require_once '../includes/functions.php';

// Admin giriş kontrolü
if (!isLoggedIn()) {
    redirectToLogin();
}

// Site ayarlarını getir
$siteSettings = getSiteSettings($conn);
$siteTitle = $siteSettings['site_title'] ?? 'Restoran Menü';
$faviconImage = $siteSettings['favicon_image'] ?? '';

$errors = [];
$success = false;

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Form verilerini al
    $name = isset($_POST['name']) ? sanitize($_POST['name']) : '';
    $description = isset($_POST['description']) ? sanitize($_POST['description']) : '';
    $sort_order = isset($_POST['sort_order']) ? intval($_POST['sort_order']) : 0;
    $status = isset($_POST['status']) ? intval($_POST['status']) : 0;
    $image_type = isset($_POST['image_type']) ? sanitize($_POST['image_type']) : 'default';
    
    // Validasyon
    if (empty($name)) {
        $errors[] = "Kategori adı gereklidir.";
    }
    
    // Resim yükleme
    $image_name = '';
    if ($image_type == 'custom' && isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
        $upload_result = uploadImage($_FILES['image']);
        
        if ($upload_result['success']) {
            $image_name = $upload_result['file_name'];
        } else {
            $errors[] = $upload_result['message'];
        }
    } elseif ($image_type == 'unsplash') {
        $image_name = isset($_POST['unsplash_url']) ? sanitize($_POST['unsplash_url']) : '';
    }
    
    // Hata yoksa veritabanına ekle
    if (empty($errors)) {
        // Slug oluştur
        $slug = slugify($name);
        
        // Veritabanına ekle
        $stmt = $conn->prepare("INSERT INTO categories (name, slug, description, image, sort_order, status) 
                               VALUES (:name, :slug, :description, :image, :sort_order, :status)");
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':slug', $slug);
        $stmt->bindParam(':description', $description);
        $stmt->bindParam(':image', $image_name);
        $stmt->bindParam(':sort_order', $sort_order, PDO::PARAM_INT);
        $stmt->bindParam(':status', $status, PDO::PARAM_INT);
        
        if ($stmt->execute()) {
            $success = true;
            $category_id = $conn->lastInsertId();
            
            // Aynı sayfada kal, URL'yi değiştirme
            // header("Location: add_category.php?success=1");
            // exit;
        } else {
            $errors[] = "Kategori eklenirken bir hata oluştu.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Yeni Kategori Ekle - <?= $siteTitle; ?></title>
    <meta name="description" content="Yeni kategori ekle - <?= $siteTitle; ?> yönetim paneli">
    <?php if (!empty($faviconImage)): ?>
        <?php if (filter_var($faviconImage, FILTER_VALIDATE_URL)): ?>
            <link rel="icon" href="<?= $faviconImage; ?>" type="image/x-icon">
            <link rel="shortcut icon" href="<?= $faviconImage; ?>" type="image/x-icon">
        <?php else: ?>
            <link rel="icon" href="../uploads/<?= $faviconImage; ?>?v=<?= time(); ?>" type="image/x-icon">
            <link rel="shortcut icon" href="../uploads/<?= $faviconImage; ?>?v=<?= time(); ?>" type="image/x-icon">
        <?php endif; ?>
    <?php else: ?>
        <link rel="icon" href="../assets/img/favicon.ico" type="image/x-icon">
        <link rel="shortcut icon" href="../assets/img/favicon.ico" type="image/x-icon">
    <?php endif; ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
    <style>
        /* Ensure file upload containers are visible when needed */
        #unsplash_url_container.show,
        #custom_image_container.show {
            display: block !important;
            visibility: visible !important;
        }
        
        /* Fix for image containers */
        .image-container {
            margin-bottom: 20px;
        }
        
        /* Fix for radio buttons */
        .form-check {
            margin-bottom: 10px;
        }
        
        /* Fix for file input */
        input[type="file"] {
            display: block;
            margin-top: 5px;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Yeni Kategori Ekle</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="categories.php" class="btn btn-sm btn-secondary">
                            <i class="fas fa-arrow-left me-1"></i> Kategorilere Dön
                        </a>
                    </div>
                </div>
                
                <?php if ($success): ?>
                    <div class="alert alert-success">
                        Kategori başarıyla eklendi. <a href="categories.php">Kategorileri görüntüle</a>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?= $error; ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <div class="card">
                    <div class="card-body">
                        <form method="post" action="add_category.php" enctype="multipart/form-data">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="name" class="form-label">Kategori Adı <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="name" name="name" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="description" class="form-label">Açıklama</label>
                                        <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="sort_order" class="form-label">Sıralama</label>
                                        <input type="number" class="form-control" id="sort_order" name="sort_order" value="0">
                                        <div class="form-text">Düşük değerler önce gösterilir.</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="status" class="form-label">Durum</label>
                                        <select class="form-select" id="status" name="status">
                                            <option value="1" selected>Aktif</option>
                                            <option value="0">Pasif</option>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label class="form-label">Kategori Resmi</label>
                                        <div class="card">
                                            <div class="card-body">
                                                <div class="mb-3 image-type-radio">
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="radio" name="image_type" id="image_type_default" value="default" checked>
                                                        <label class="form-check-label" for="image_type_default">
                                                            Varsayılan resim kullan
                                                        </label>
                                                    </div>
                                                    <div class="form-text">Sistem kategori adına göre otomatik bir resim seçecektir.</div>
                                                </div>
                                                
                                                <div class="mb-3 image-type-radio">
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="radio" name="image_type" id="image_type_unsplash" value="unsplash">
                                                        <label class="form-check-label" for="image_type_unsplash">
                                                            Unsplash resmi kullan
                                                        </label>
                                                    </div>
                                                    <div id="unsplash_url_container" class="mt-2 d-none">
                                                        <input type="text" class="form-control" id="unsplash_url" name="unsplash_url" placeholder="Unsplash resim URL'si">
                                                        <div class="form-text">Örnek: https://images.unsplash.com/photo-1546241072-48010ad2862c</div>
                                                    </div>
                                                </div>
                                                
                                                <div class="mb-3 image-type-radio">
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="radio" name="image_type" id="image_type_custom" value="custom">
                                                        <label class="form-check-label" for="image_type_custom">
                                                            Özel resim yükle
                                                        </label>
                                                    </div>
                                                    <div id="custom_image_container" class="mt-2 d-none">
                                                        <input type="file" class="form-control" id="image" name="image" accept="image/*">
                                                        <div class="form-text">Maksimum dosya boyutu: 2MB. İzin verilen formatlar: JPG, JPEG, PNG, GIF</div>
                                                        <div id="image-preview" class="mt-2"></div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">Kategori Ekle</button>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="../assets/js/admin.js"></script>
    <script>
        // Resim seçeneklerini göster/gizle
        document.addEventListener('DOMContentLoaded', function() {
            const imageTypeRadios = document.querySelectorAll('input[name="image_type"]');
            const unsplashUrlContainer = document.getElementById('unsplash_url_container');
            const customImageContainer = document.getElementById('custom_image_container');
            
            // Sayfa yüklendiğinde seçili olan radio butonuna göre konteynerları göster/gizle
            function updateContainers() {
                const selectedValue = document.querySelector('input[name="image_type"]:checked').value;
                
                if (selectedValue === 'unsplash') {
                    unsplashUrlContainer.classList.remove('d-none');
                    unsplashUrlContainer.classList.add('show');
                    customImageContainer.classList.add('d-none');
                    customImageContainer.classList.remove('show');
                } else if (selectedValue === 'custom') {
                    unsplashUrlContainer.classList.add('d-none');
                    unsplashUrlContainer.classList.remove('show');
                    customImageContainer.classList.remove('d-none');
                    customImageContainer.classList.add('show');
                } else {
                    unsplashUrlContainer.classList.add('d-none');
                    unsplashUrlContainer.classList.remove('show');
                    customImageContainer.classList.add('d-none');
                    customImageContainer.classList.remove('show');
                }
            }
            
            // İlk yüklemede kontrol et
            updateContainers();
            
            // Radio butonlarına change event listener ekle
            imageTypeRadios.forEach(radio => {
                radio.addEventListener('change', updateContainers);
            });
            
            // Resim önizleme
            const imageInput = document.getElementById('image');
            if (imageInput) {
                imageInput.addEventListener('change', function(e) {
                    const file = e.target.files[0];
                    if (file) {
                        const reader = new FileReader();
                        reader.onload = function(e) {
                            const preview = document.getElementById('image-preview');
                            if (preview) {
                                preview.innerHTML = `<img src="${e.target.result}" class="img-thumbnail" style="max-width: 200px;">`;
                            }
                        }
                        reader.readAsDataURL(file);
                    }
                });
            }
            
            // Force update after a short delay to ensure DOM is fully loaded
            setTimeout(updateContainers, 100);
        });
    </script>
</body>
</html>