<?php
session_start();
require_once '../config/db.php';
require_once '../includes/functions.php';

// Admin giriş kontrolü
if (!isLoggedIn()) {
    redirectToLogin();
}

// Site ayarlarını getir
$siteSettings = getSiteSettings($conn);
$siteTitle = $siteSettings['site_title'] ?? 'Restoran Menü';
$faviconImage = $siteSettings['favicon_image'] ?? '';

// Kategorileri getir
$stmt = $conn->prepare("SELECT * FROM categories WHERE status = 1 ORDER BY sort_order ASC");
$stmt->execute();
$categories = $stmt->fetchAll();

$errors = [];
$success = false;

// Form gönderildiğinde
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Form verilerini al
    $name = isset($_POST['name']) ? sanitize($_POST['name']) : '';
    $category_id = isset($_POST['category_id']) ? intval($_POST['category_id']) : 0;
    $description = isset($_POST['description']) ? sanitize($_POST['description']) : '';
    $price = isset($_POST['price']) ? floatval($_POST['price']) : 0;
    $sort_order = isset($_POST['sort_order']) ? intval($_POST['sort_order']) : 0;
    $status = isset($_POST['status']) ? intval($_POST['status']) : 0;
    $featured = isset($_POST['featured']) ? intval($_POST['featured']) : 0;
    
    // Validasyon
    if (empty($name)) {
        $errors[] = "Ürün adı gereklidir.";
    }
    
    if ($category_id <= 0) {
        $errors[] = "Geçerli bir kategori seçmelisiniz.";
    }
    
    if ($price <= 0) {
        $errors[] = "Fiyat sıfırdan büyük olmalıdır.";
    }
    
    // Resim yükleme
    $image_name = '';
    if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
        $upload_result = uploadImage($_FILES['image']);
        
        if ($upload_result['success']) {
            $image_name = $upload_result['file_name'];
        } else {
            $errors[] = $upload_result['message'];
        }
    }
    
    // Hata yoksa veritabanına ekle
    if (empty($errors)) {
        // Slug oluştur
        $slug = slugify($name);
        
        // Veritabanına ekle
        $stmt = $conn->prepare("INSERT INTO products (category_id, name, slug, description, price, image, sort_order, status, featured) 
                               VALUES (:category_id, :name, :slug, :description, :price, :image, :sort_order, :status, :featured)");
        $stmt->bindParam(':category_id', $category_id, PDO::PARAM_INT);
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':slug', $slug);
        $stmt->bindParam(':description', $description);
        $stmt->bindParam(':price', $price);
        $stmt->bindParam(':image', $image_name);
        $stmt->bindParam(':sort_order', $sort_order, PDO::PARAM_INT);
        $stmt->bindParam(':status', $status, PDO::PARAM_INT);
        $stmt->bindParam(':featured', $featured, PDO::PARAM_INT);
        
        if ($stmt->execute()) {
            $success = true;
            $product_id = $conn->lastInsertId();
            
            // Aynı sayfada kal, URL'yi değiştirme
            // header("Location: add_product.php?success=1");
            // exit;
        } else {
            $errors[] = "Ürün eklenirken bir hata oluştu.";
            
            // Resim yüklendiyse sil
            if (!empty($image_name)) {
                $image_path = "../uploads/" . $image_name;
                if (file_exists($image_path)) {
                    unlink($image_path);
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Yeni Ürün Ekle - <?= $siteTitle; ?></title>
    <meta name="description" content="Yeni ürün ekle - <?= $siteTitle; ?> yönetim paneli">
    <?php if (!empty($faviconImage)): ?>
        <?php if (filter_var($faviconImage, FILTER_VALIDATE_URL)): ?>
            <link rel="icon" href="<?= $faviconImage; ?>" type="image/x-icon">
            <link rel="shortcut icon" href="<?= $faviconImage; ?>" type="image/x-icon">
        <?php else: ?>
            <link rel="icon" href="../uploads/<?= $faviconImage; ?>?v=<?= time(); ?>" type="image/x-icon">
            <link rel="shortcut icon" href="../uploads/<?= $faviconImage; ?>?v=<?= time(); ?>" type="image/x-icon">
        <?php endif; ?>
    <?php else: ?>
        <link rel="icon" href="../assets/img/favicon.ico" type="image/x-icon">
        <link rel="shortcut icon" href="../assets/img/favicon.ico" type="image/x-icon">
    <?php endif; ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Yeni Ürün Ekle</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="products.php" class="btn btn-sm btn-secondary">
                            <i class="fas fa-arrow-left me-1"></i> Ürünlere Dön
                        </a>
                    </div>
                </div>
                
                <?php if ($success): ?>
                    <div class="alert alert-success">
                        Ürün başarıyla eklendi. <a href="products.php">Ürünleri görüntüle</a>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?> <li><?= $error; ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <div class="card">
                    <div class="card-body">
                        <form method="post" action="add_product.php" enctype="multipart/form-data">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="name" class="form-label">Ürün Adı <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="name" name="name" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="category_id" class="form-label">Kategori <span class="text-danger">*</span></label>
                                        <select class="form-select" id="category_id" name="category_id" required>
                                            <option value="">Kategori Seçin</option>
                                            <?php foreach ($categories as $category): ?>
                                                <option value="<?= $category['id']; ?>"><?= $category['name']; ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="price" class="form-label">Fiyat (₺) <span class="text-danger">*</span></label>
                                        <input type="number" class="form-control" id="price" name="price" step="0.01" min="0" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="image" class="form-label">Ürün Resmi</label>
                                        <input type="file" class="form-control" id="image" name="image" accept="image/*">
                                        <div class="form-text">Maksimum dosya boyutu: 2MB. İzin verilen formatlar: JPG, JPEG, PNG, GIF</div>
                                        <div id="image-preview" class="mt-2"></div>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="description" class="form-label">Açıklama</label>
                                        <textarea class="form-control" id="description" name="description" rows="5"></textarea>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="sort_order" class="form-label">Sıralama</label>
                                        <input type="number" class="form-control" id="sort_order" name="sort_order" value="0">
                                        <div class="form-text">Düşük değerler önce gösterilir.</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="status" class="form-label">Durum</label>
                                        <select class="form-select" id="status" name="status">
                                            <option value="1" selected>Aktif</option>
                                            <option value="0">Pasif</option>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="featured" class="form-label">Öne Çıkan</label>
                                        <select class="form-select" id="featured" name="featured">
                                            <option value="1">Evet</option>
                                            <option value="0" selected>Hayır</option>
                                        </select>
                                        <div class="form-text">Öne çıkan ürünler ana sayfada gösterilir.</div>
                                    </div>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">Ürün Ekle</button>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="../assets/js/admin.js"></script>
    <script>
        // Resim önizleme
        document.getElementById('image').addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const preview = document.getElementById('image-preview');
                    preview.innerHTML = `<img src="${e.target.result}" class="img-thumbnail" style="max-width: 200px;">`;
                }
                reader.readAsDataURL(file);
            }
        });
    </script>
</body>
</html>