// Ana JavaScript Dosyası
document.addEventListener('DOMContentLoaded', function() {
    // Sayfa yüklendiğinde çalışacak kodlar
    console.log('Restoran Menü Sistemi yüklendi');
    
    // Ürün kartları için hover efekti
    const productCards = document.querySelectorAll('.product-card, .featured-card');
    
    productCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-5px)';
            this.style.boxShadow = '0 10px 20px rgba(0, 0, 0, 0.1)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
            this.style.boxShadow = '0 5px 15px rgba(0, 0, 0, 0.05)';
        });
    });
    
    // Ana Slider için
    const sliderItems = document.querySelectorAll('.slider-item');
    const sliderDots = document.querySelectorAll('.slider-dot');
    const sliderPrev = document.getElementById('sliderPrev');
    const sliderNext = document.getElementById('sliderNext');
    
    if (sliderItems.length > 1) {
        let currentSlide = 0;
        let slideInterval;
        let isPaused = false;
        
        // Slider'ı göster
        function showSlide(index) {
            // Tüm sliderları gizle
            sliderItems.forEach(item => {
                item.classList.remove('active');
            });
            
            // Tüm noktaları pasif yap
            sliderDots.forEach(dot => {
                dot.classList.remove('active');
            });
            
            // Aktif slider'ı göster
            sliderItems[index].classList.add('active');
            
            // Aktif noktayı işaretle
            if (sliderDots[index]) {
                sliderDots[index].classList.add('active');
            }
            
            // Geçerli slide'ı güncelle
            currentSlide = index;
        }
        
        // Sonraki slide'a geç
        function nextSlide() {
            if (isPaused) return;
            
            let next = currentSlide + 1;
            if (next >= sliderItems.length) {
                next = 0;
            }
            showSlide(next);
        }
        
        // Önceki slide'a geç
        function prevSlide() {
            if (isPaused) return;
            
            let prev = currentSlide - 1;
            if (prev < 0) {
                prev = sliderItems.length - 1;
            }
            showSlide(prev);
        }
        
        // Otomatik geçişi başlat
        function startSlideInterval() {
            if (document.querySelector('.slider-autoplay') && document.querySelector('.slider-autoplay').value === '1') {
                const interval = parseInt(document.querySelector('.slider-interval')?.value || '5000');
                slideInterval = setInterval(nextSlide, interval);
            }
        }
        
        // Otomatik geçişi durdur
        function stopSlideInterval() {
            clearInterval(slideInterval);
        }
        
        // İlk slide'ı göster
        showSlide(0);
        
        // Otomatik geçişi başlat
        startSlideInterval();
        
        // Noktalara tıklama olayı
        sliderDots.forEach((dot, index) => {
            dot.addEventListener('click', () => {
                isPaused = true;
                stopSlideInterval();
                showSlide(index);
                
                // Kısa bir süre sonra otomatik geçişi tekrar başlat
                setTimeout(() => {
                    isPaused = false;
                    startSlideInterval();
                }, 5000);
            });
        });
        
        // Önceki/sonraki butonları
        if (sliderPrev && sliderNext) {
            sliderPrev.addEventListener('click', () => {
                isPaused = true;
                stopSlideInterval();
                prevSlide();
                
                // Kısa bir süre sonra otomatik geçişi tekrar başlat
                setTimeout(() => {
                    isPaused = false;
                    startSlideInterval();
                }, 5000);
            });
            
            sliderNext.addEventListener('click', () => {
                isPaused = true;
                stopSlideInterval();
                nextSlide();
                
                // Kısa bir süre sonra otomatik geçişi tekrar başlat
                setTimeout(() => {
                    isPaused = false;
                    startSlideInterval();
                }, 5000);
            });
        }
        
        // Slider üzerine gelince durdur
        const sliderWrapper = document.querySelector('.slider-wrapper');
        if (sliderWrapper) {
            sliderWrapper.addEventListener('mouseenter', () => {
                isPaused = true;
                stopSlideInterval();
            });
            
            sliderWrapper.addEventListener('mouseleave', () => {
                isPaused = false;
                startSlideInterval();
            });
            
            // Dokunma olayları için
            sliderWrapper.addEventListener('touchstart', () => {
                isPaused = true;
                stopSlideInterval();
            });
            
            sliderWrapper.addEventListener('touchend', () => {
                setTimeout(() => {
                    isPaused = false;
                    startSlideInterval();
                }, 5000);
            });
        }
    }
    
    // Öne çıkan ürünler için kaydırma kontrolü
    const featuredSlider = document.getElementById('featuredSlider');
    const prevBtn = document.getElementById('prevBtn');
    const nextBtn = document.getElementById('nextBtn');
    
    if (featuredSlider && prevBtn && nextBtn) {
        let scrollAmount = 0;
        let isPaused = false;
        let scrollInterval;
        let autoScrollEnabled = document.querySelector('.slider-autoplay')?.value === '1';
        
        // Otomatik kaydırma başlat
        function startScroll() {
            if (scrollInterval) clearInterval(scrollInterval);
            
            if (!autoScrollEnabled) return;
            
            scrollInterval = setInterval(function() {
                if (isPaused) return;
                
                // Sınıra ulaşıldığında başa dön
                if (scrollAmount >= featuredSlider.scrollWidth - featuredSlider.clientWidth - 10) {
                    scrollAmount = 0;
                    featuredSlider.scrollTo({
                        left: 0,
                        behavior: 'smooth'
                    });
                } else {
                    // Yavaşça kaydır
                    scrollAmount += 1;
                    featuredSlider.scrollLeft = scrollAmount;
                }
            }, 30);
        }
        
        // Kaydırmayı durdur
        function stopScroll() {
            if (scrollInterval) {
                clearInterval(scrollInterval);
                scrollInterval = null;
            }
        }
        
        // Başlangıçta kaydırmayı başlat
        startScroll();
        
        // Fare üzerine geldiğinde durdur
        featuredSlider.addEventListener('mouseenter', function() {
            isPaused = true;
            stopScroll();
        });
        
        featuredSlider.addEventListener('mouseleave', function() {
            isPaused = false;
            startScroll();
        });
        
        // Dokunma olayları için
        featuredSlider.addEventListener('touchstart', function() {
            isPaused = true;
            stopScroll();
        });
        
        featuredSlider.addEventListener('touchend', function() {
            setTimeout(function() {
                isPaused = false;
                startScroll();
            }, 3000);
        });
        
        // Önceki butonu
        prevBtn.addEventListener('click', function() {
            isPaused = true;
            stopScroll();
            
            // Bir önceki öğeye kaydır
            const itemWidth = featuredSlider.querySelector('.col-6')?.offsetWidth || 200;
            scrollAmount = Math.max(0, scrollAmount - itemWidth * 2);
            featuredSlider.scrollTo({
                left: scrollAmount,
                behavior: 'smooth'
            });
            
            setTimeout(function() {
                isPaused = false;
                startScroll();
            }, 3000);
        });
        
        // Sonraki butonu
        nextBtn.addEventListener('click', function() {
            isPaused = true;
            stopScroll();
            
            // Bir sonraki öğeye kaydır
            const itemWidth = featuredSlider.querySelector('.col-6')?.offsetWidth || 200;
            const maxScroll = featuredSlider.scrollWidth - featuredSlider.clientWidth;
            scrollAmount = Math.min(maxScroll, scrollAmount + itemWidth * 2);
            featuredSlider.scrollTo({
                left: scrollAmount,
                behavior: 'smooth'
            });
            
            setTimeout(function() {
                isPaused = false;
                startScroll();
            }, 3000);
        });
        
        // Kaydırma olayını dinle
        featuredSlider.addEventListener('scroll', function() {
            // Kullanıcı manuel olarak kaydırdığında, otomatik kaydırmayı güncelle
            scrollAmount = featuredSlider.scrollLeft;
        });
    }
    
    // Mobil menü için
    const navbarToggler = document.querySelector('.navbar-toggler');
    if (navbarToggler) {
        navbarToggler.addEventListener('click', function() {
            document.querySelector('.navbar-collapse').classList.toggle('show');
        });
    }
    
    // Resim yükleme önizleme
    const imageInputs = document.querySelectorAll('input[type="file"][accept*="image"]');
    
    imageInputs.forEach(input => {
        input.addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                const reader = new FileReader();
                
                reader.onload = function(e) {
                    const preview = document.createElement('img');
                    preview.src = e.target.result;
                    preview.className = 'img-thumbnail mt-2';
                    preview.style.maxWidth = '200px';
                    
                    const previewContainer = input.parentElement.querySelector('.preview-container');
                    
                    if (previewContainer) {
                        previewContainer.innerHTML = '';
                        previewContainer.appendChild(preview);
                    } else {
                        const newPreviewContainer = document.createElement('div');
                        newPreviewContainer.className = 'preview-container';
                        newPreviewContainer.appendChild(preview);
                        input.parentElement.appendChild(newPreviewContainer);
                    }
                };
                
                reader.readAsDataURL(file);
            }
        });
    });
    
    // Türkçe karakter desteği için
    const slugifyInputs = document.querySelectorAll('input[data-slugify]');
    if (slugifyInputs.length > 0) {
        slugifyInputs.forEach(input => {
            const targetId = input.dataset.slugify;
            const targetInput = document.getElementById(targetId);
            
            if (targetInput) {
                input.addEventListener('input', function() {
                    const slug = slugify(this.value);
                    targetInput.value = slug;
                });
            }
        });
    }
    
    // Slug oluşturma fonksiyonu
    function slugify(text) {
        // Türkçe karakterleri değiştir
        const turkishChars = {
            'ı': 'i', 'ğ': 'g', 'ü': 'u', 'ş': 's', 'ö': 'o', 'ç': 'c',
            'İ': 'i', 'Ğ': 'g', 'Ü': 'u', 'Ş': 's', 'Ö': 'o', 'Ç': 'c'
        };
        
        return text
            .toString()
            .toLowerCase()
            .replace(/[ıİğĞüÜşŞöÖçÇ]/g, function(letter) {
                return turkishChars[letter] || letter;
            })
            .replace(/\s+/g, '-')           // Boşlukları tire ile değiştir
            .replace(/[^\w\-]+/g, '')       // Alfanumerik olmayan karakterleri kaldır
            .replace(/\-\-+/g, '-')         // Birden fazla tireyi tek tireye dönüştür
            .replace(/^-+/, '')             // Baştaki tireleri kaldır
            .replace(/-+$/, '');            // Sondaki tireleri kaldır
    }
    
    // Ürün detay modalları için
    const productModals = document.querySelectorAll('.modal');
    if (productModals.length > 0) {
        productModals.forEach(modal => {
            modal.addEventListener('show.bs.modal', function() {
                // Modal açıldığında yapılacak işlemler
                const modalImage = this.querySelector('.modal-body img');
                if (modalImage) {
                    modalImage.style.opacity = '0';
                    setTimeout(() => {
                        modalImage.style.opacity = '1';
                        modalImage.style.transition = 'opacity 0.3s ease-in-out';
                    }, 100);
                }
            });
        });
    }
    
    // Sayfa yüklendiğinde animasyon efektleri
    const animateOnLoad = () => {
        // Kategori kartları için animasyon
        const categoryCards = document.querySelectorAll('.category-card');
        categoryCards.forEach((card, index) => {
            setTimeout(() => {
                card.style.opacity = '1';
                card.style.transform = 'translateY(0)';
            }, 100 * index);
        });
        
        // Ürün kartları için animasyon
        const productCards = document.querySelectorAll('.product-card');
        productCards.forEach((card, index) => {
            setTimeout(() => {
                card.style.opacity = '1';
                card.style.transform = 'translateY(0)';
            }, 150 * index);
        });
    };
    
    // Sayfa tamamen yüklendiğinde animasyonları başlat
    window.addEventListener('load', animateOnLoad);
    
    // Fiyat formatı için
    const formatPrice = (price) => {
        return new Intl.NumberFormat('tr-TR', {
            style: 'currency',
            currency: 'TRY',
            minimumFractionDigits: 2
        }).format(price);
    };
    
    // Sayfa kaydırma animasyonu
    const smoothScroll = (target, duration = 500) => {
        const targetElement = document.querySelector(target);
        if (!targetElement) return;
        
        const targetPosition = targetElement.getBoundingClientRect().top + window.pageYOffset;
        const startPosition = window.pageYOffset;
        const distance = targetPosition - startPosition;
        let startTime = null;
        
        const animation = (currentTime) => {
            if (startTime === null) startTime = currentTime;
            const timeElapsed = currentTime - startTime;
            const run = ease(timeElapsed, startPosition, distance, duration);
            window.scrollTo(0, run);
            if (timeElapsed < duration) requestAnimationFrame(animation);
        };
        
        // Easing fonksiyonu
        const ease = (t, b, c, d) => {
            t /= d / 2;
            if (t < 1) return c / 2 * t * t + b;
            t--;
            return -c / 2 * (t * (t - 2) - 1) + b;
        };
        
        requestAnimationFrame(animation);
    };
    
    // Sayfa içi bağlantılar için smooth scroll
    const pageLinks = document.querySelectorAll('a[href^="#"]');
    pageLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            if (this.getAttribute('href') !== '#') {
                e.preventDefault();
                const target = this.getAttribute('href');
                smoothScroll(target);
            }
        });
    });
    
    // Sayfa yüklendiğinde URL hash varsa o bölüme kaydır
    if (window.location.hash) {
        setTimeout(() => {
            smoothScroll(window.location.hash);
        }, 500);
    }
    
    // Lazy loading için
    if ('IntersectionObserver' in window) {
        const lazyImages = document.querySelectorAll('img[data-src]');
        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    img.src = img.dataset.src;
                    img.removeAttribute('data-src');
                    imageObserver.unobserve(img);
                }
            });
        });
        
        lazyImages.forEach(img => {
            imageObserver.observe(img);
        });
    }
    
    // Sayfa yükleme performansı için
    window.addEventListener('load', () => {
        // Sayfa yükleme süresini ölç
        const loadTime = window.performance.timing.domContentLoadedEventEnd - window.performance.timing.navigationStart;
        console.log(`Sayfa ${loadTime}ms içinde yüklendi`);
    });
});