<?php
// LISANS_KODU_BURADA
// Kategorileri getir
function getCategories($conn) {
    try {
        $stmt = $conn->prepare("SELECT * FROM categories WHERE status = 1 ORDER BY sort_order ASC");
        $stmt->execute();
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log("Kategori getirme hatası: " . $e->getMessage());
        return [];
    }
}

// Kategori ID'sine göre ürünleri getir
function getProductsByCategory($conn, $category_id) {
    try {
        $stmt = $conn->prepare("SELECT * FROM products WHERE category_id = :category_id AND status = 1 ORDER BY sort_order ASC");
        $stmt->bindParam(':category_id', $category_id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log("Ürün getirme hatası: " . $e->getMessage());
        return [];
    }
}

// Tüm ürünleri getir
function getAllProducts($conn) {
    try {
        $stmt = $conn->prepare("SELECT p.*, c.name as category_name FROM products p 
                               LEFT JOIN categories c ON p.category_id = c.id 
                               WHERE p.status = 1 
                               ORDER BY p.sort_order ASC");
        $stmt->execute();
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log("Tüm ürünleri getirme hatası: " . $e->getMessage());
        return [];
    }
}

// Öne çıkan ürünleri getir
function getFeaturedProducts($conn) {
    try {
        $stmt = $conn->prepare("SELECT p.*, c.name as category_name FROM products p 
                               LEFT JOIN categories c ON p.category_id = c.id 
                               WHERE p.status = 1 AND p.featured = 1
                               ORDER BY p.sort_order ASC");
        $stmt->execute();
        $featuredProducts = $stmt->fetchAll();
        
        // Eğer öne çıkan ürün yoksa, her kategoriden bir ürün göster
        if (count($featuredProducts) == 0) {
            $categories = getCategories($conn);
            $featuredProducts = [];
            
            foreach ($categories as $category) {
                $stmt = $conn->prepare("SELECT p.*, c.name as category_name FROM products p 
                                       LEFT JOIN categories c ON p.category_id = c.id 
                                       WHERE p.category_id = :category_id AND p.status = 1 
                                       ORDER BY p.sort_order ASC LIMIT 1");
                $stmt->bindParam(':category_id', $category['id'], PDO::PARAM_INT);
                $stmt->execute();
                $product = $stmt->fetch();
                
                if ($product) {
                    $featuredProducts[] = $product;
                }
            }
        }
        
        return $featuredProducts;
    } catch (PDOException $e) {
        error_log("Öne çıkan ürünleri getirme hatası: " . $e->getMessage());
        return [];
    }
}

// ID'ye göre ürün getir
function getProductById($conn, $id) {
    try {
        $stmt = $conn->prepare("SELECT * FROM products WHERE id = :id");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch();
    } catch (PDOException $e) {
        error_log("Ürün getirme hatası: " . $e->getMessage());
        return false;
    }
}

// ID'ye göre kategori getir
function getCategoryById($conn, $id) {
    try {
        $stmt = $conn->prepare("SELECT * FROM categories WHERE id = :id");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch();
    } catch (PDOException $e) {
        error_log("Kategori getirme hatası: " . $e->getMessage());
        return false;
    }
}

// String'i slug formatına dönüştür
function slugify($text) {
    // Türkçe karakterleri değiştir
    $text = str_replace(
        ['ı', 'ğ', 'ü', 'ş', 'ö', 'ç', 'İ', 'Ğ', 'Ü', 'Ş', 'Ö', 'Ç', '&'],
        ['i', 'g', 'u', 's', 'o', 'c', 'i', 'g', 'u', 's', 'o', 'c', 've'],
        $text
    );
    
    // Küçük harfe dönüştür
    $text = mb_strtolower($text, 'UTF-8');
    
    // Alfanumerik olmayan karakterleri tire ile değiştir
    $text = preg_replace('/[^a-z0-9\-]/', '-', $text);
    
    // Birden fazla tireyi tek tireye dönüştür
    $text = preg_replace('/-+/', '-', $text);
    
    // Baştaki ve sondaki tireleri kaldır
    return trim($text, '-');
}

// Güvenli input temizleme
function sanitize($input) {
    if (is_array($input)) {
        foreach ($input as $key => $value) {
            $input[$key] = sanitize($value);
        }
        return $input;
    }
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

// XSS korumalı çıktı
function safeOutput($text) {
    return htmlspecialchars($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
}

// Resim yükleme fonksiyonu
function uploadImage($file) {
    $target_dir = "../uploads/";
    
    // Klasör yoksa oluştur
    if (!file_exists($target_dir)) {
        if (!mkdir($target_dir, 0755, true)) {
            return ["success" => false, "message" => "Yükleme klasörü oluşturulamadı."];
        }
    }
    
    // Klasör yazılabilir mi kontrol et
    if (!is_writable($target_dir)) {
        return ["success" => false, "message" => "Yükleme klasörü yazılabilir değil."];
    }
    
    $file_name = basename($file["name"]);
    $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
    
    // Benzersiz dosya adı oluştur - Daha güvenilir benzersiz isim için microtime ekle
    $new_file_name = uniqid(microtime(true) . '_') . '.' . $file_ext;
    $target_file = $target_dir . $new_file_name;
    
    // Dosya türünü kontrol et
    $allowed_types = ["jpg", "jpeg", "png", "gif", "ico", "svg"];
    if (!in_array($file_ext, $allowed_types)) {
        return ["success" => false, "message" => "Sadece JPG, JPEG, PNG, GIF, ICO ve SVG dosyaları yüklenebilir."];
    }
    
    // Dosya boyutunu kontrol et (2MB)
    if ($file["size"] > 2000000) {
        return ["success" => false, "message" => "Dosya boyutu 2MB'dan büyük olamaz."];
    }
    
    // Dosya gerçekten bir resim mi kontrol et
    $check = getimagesize($file["tmp_name"]);
    if ($check === false && $file_ext !== "svg") {
        return ["success" => false, "message" => "Yüklenen dosya bir resim değil."];
    }
    
    // Dosyayı yükle
    if (move_uploaded_file($file["tmp_name"], $target_file)) {
        // Dosya izinlerini güvenli hale getir
        chmod($target_file, 0644);
        return ["success" => true, "file_name" => $new_file_name];
    } else {
        return ["success" => false, "message" => "Dosya yüklenirken bir hata oluştu."];
    }
}

// Admin giriş kontrolü
function isLoggedIn() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

// Admin giriş sayfasına yönlendir
function redirectToLogin() {
    header("Location: login.php");
    exit;
}

// CSRF token oluştur
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// CSRF token doğrula
function validateCSRFToken($token) {
    if (!isset($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
        return false;
    }
    return true;
}

// Site ayarlarını getir
function getSiteSettings($conn) {
    try {
        $stmt = $conn->prepare("SELECT setting_key, setting_value FROM settings");
        $stmt->execute();
        $settings_array = $stmt->fetchAll();
        
        // Dizi formatına dönüştür
        $settings = [];
        foreach ($settings_array as $setting) {
            $settings[$setting['setting_key']] = $setting['setting_value'];
        }
        
        return $settings;
    } catch (PDOException $e) {
        error_log("Ayarları getirme hatası: " . $e->getMessage());
        return [];
    }
}

// Ayar güncelle veya ekle
function updateSetting($conn, $key, $value) {
    try {
        // Ayarın varlığını kontrol et
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM settings WHERE setting_key = :key");
        $stmt->bindParam(':key', $key);
        $stmt->execute();
        $result = $stmt->fetch();
        
        if ($result['count'] > 0) {
            // Ayar varsa güncelle
            $stmt = $conn->prepare("UPDATE settings SET setting_value = :value WHERE setting_key = :key");
        } else {
            // Ayar yoksa ekle
            $stmt = $conn->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (:key , :value)");
        }
        
        $stmt->bindParam(':key', $key);
        $stmt->bindParam(':value', $value);
        return $stmt->execute();
    } catch (PDOException $e) {
        error_log("Ayar güncelleme hatası: " . $e->getMessage());
        return false;
    }
}

// SEO dostu URL oluştur
function createSeoUrl($type, $id, $slug) {
    if ($type == 'category') {
        return "kategori/$id-$slug.html";
    } elseif ($type == 'product') {
        return "urun/$id-$slug.html";
    }
    return "#";
}

// SEO dostu URL'den ID ve slug çıkar
function parseSeoUrl($url) {
    $result = ['id' => 0, 'slug' => ''];
    
    // kategori/123-ornek-kategori.html veya urun/123-ornek-urun.html formatını kontrol et
    if (preg_match('/^(kategori|urun)\/(\d+)-([a-z0-9\-]+)\.html$/', $url, $matches)) {
        $result['type'] = ($matches[1] == 'kategori') ? 'category' : 'product';
        $result['id'] = intval($matches[2]);
        $result['slug'] = $matches[3];
    }
    
    return $result;
}

// Kategori için varsayılan resim URL'si al
function getCategoryDefaultImage($category_name) {
    $categoryImages = [
        'Başlangıçlar' => 'https://images.unsplash.com/photo-1546241072-48010ad2862c?q=80&w=500',
        'Ana Yemekler' => 'https://images.unsplash.com/photo-1544025162-d76694265947?q=80&w=500',
        'Tatlılar' => 'https://images.unsplash.com/photo-1563805042-7684c019e1cb?q=80&w=500',
        'İçecekler' => 'https://images.unsplash.com/photo-1544145945-f90425340c7e?q=80&w=500',
        'Pideler' => 'https://images.unsplash.com/photo-1513104890138-7c749659a591?q=80&w=500',
        'Kebaplar' => 'https://images.unsplash.com/photo-1544025162-d76694265947?q=80&w=500',
        'Dönerler' => 'https://images.unsplash.com/photo-1633321702518-7feccafb94d5?q=80&w=500',
        'Lahmacunlar' => 'https://images.unsplash.com/photo-1632934626483-2e1c1a1b3d27?q=80&w=500',
        'Salatalar' => 'https://images.unsplash.com/photo-1512621776951-a57141f2eefd?q=80&w=500'
    ];
    
    return isset($categoryImages[$category_name]) ? 
        $categoryImages[$category_name] : 
        'https://images.unsplash.com/photo-1555939594-58d7cb561ad1?q=80&w=500';
}